/*
 * @(#) RedBlackTree.java Feb 16, 2004
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.dsol.eventlists;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import nl.tudelft.simulation.dsol.formalisms.devs.SimEventInterface;

/**
 * A RedBlackTree implementation of the eventlistInterface. This implementation
 * is based on Java's TreeSet.
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.4 2004-03-28
 * @since 1.0
 */
public class RedBlackTree implements EventListInterface
{
	/** the counter of events added to this list */
	private static long counter = 0L;

	/** RED */
	protected static final boolean RED = false;

	/** BLACK */
	protected static final boolean BLACK = true;

	/** the root of the tree */
	protected transient Entry root = null;

	/** The number of entries in the tree */
	protected transient int size = 0;

	/** The number of structural modifications to the tree. */
	protected transient int modCount = 0;

	/**
	 * constructs a new RedBlackTree
	 */
	public RedBlackTree()
	{
		super();
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface
	 *      #add(nl.tudelft.simulation.dsol.formalisms.devs.SimEventInterface)
	 */
	public synchronized boolean add(final SimEventInterface event)
	{
		event.setID(RedBlackTree.counter++);
		Entry entry = this.root;
		if (entry == null)
		{
			this.incrementSize();
			this.root = new Entry(event, null);
			return true;
		}
		while (true)
		{
			int cmp = event.compareTo(entry.simEvent);
			if (cmp == 0)
			{
				entry.simEvent = event;
				return false;
			} else if (cmp < 0)
			{
				if (entry.left != null)
				{
					entry = entry.left;
				} else
				{
					this.incrementSize();
					entry.left = new Entry(event, entry);
					fixAfterInsertion(entry.left);
					return true;
				}
			} else
			{ // cmp > 0
				if (entry.right != null)
				{
					entry = entry.right;
				} else
				{
					incrementSize();
					entry.right = new Entry(event, entry);
					fixAfterInsertion(entry.right);
					return true;
				}
			}
		}
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface
	 *      #addAll(Collection)
	 */
	public synchronized boolean addAll(final Collection collection)
	{
		for (Iterator iterator = collection.iterator(); iterator.hasNext();)
		{
			if (!this.add((SimEventInterface) iterator.next()))
			{
				return false;
			}
		}
		return true;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface#clear()
	 */
	public synchronized void clear()
	{
		this.modCount++;
		this.size = 0;
		this.root = null;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface
	 *      #contains(nl.tudelft.simulation.dsol.formalisms.devs.SimEventInterface)
	 */
	public synchronized boolean contains(final SimEventInterface event)
	{
		if (this.root == null)
		{
			return false;
		}
		if (event == null)
		{
			return this.eventSearchNull(this.root);
		}
		return this.eventSearchNonNull(this.root, event);
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface
	 *      #containsAll(Collection)
	 */
	public synchronized boolean containsAll(final Collection collection)
	{
		for (Iterator iterator = collection.iterator(); iterator.hasNext();)
		{
			if (!this.contains((SimEventInterface) iterator.next()))
			{
				return false;
			}
		}
		return true;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface #first()
	 */
	public synchronized SimEventInterface first()
	{
		Entry first = this.firstEntry();
		if (first == null)
		{
			return null;
		}
		return first.simEvent;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface#isEmpty()
	 */
	public synchronized boolean isEmpty()
	{
		if (this.size == 0)
		{
			return true;
		}
		return false;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface #iterator()
	 */
	public synchronized Iterator iterator()
	{
		return new EventListIterator();
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface#last()
	 */
	public synchronized SimEventInterface last()
	{
		Entry last = this.lastEntry();
		if (last == null)
		{
			return null;
		}
		return last.simEvent;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface
	 *      #remove(nl.tudelft.simulation.dsol.formalisms.devs.SimEventInterface)
	 */
	public synchronized boolean remove(final SimEventInterface event)
	{
		Entry entry = getEntry(event);
		if (entry == null)
		{
			return false;
		}
		this.deleteEntry(entry);
		return true;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface#removeAll(Collection)
	 */
	public synchronized boolean removeAll(final Collection collection)
	{
		for (Iterator iterator = collection.iterator(); iterator.hasNext();)
		{
			if (!this.remove((SimEventInterface) iterator.next()))
			{
				return false;
			}
		}
		return true;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface
	 *      #removeFirst()
	 */
	public synchronized SimEventInterface removeFirst()
	{
		Entry entry = this.firstEntry();
		this.deleteEntry(entry);
		return entry.simEvent;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface#removeLast()
	 */
	public synchronized SimEventInterface removeLast()
	{
		Entry entry = this.lastEntry();
		this.deleteEntry(entry);
		return entry.simEvent;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface#size()
	 */
	public synchronized int size()
	{
		return this.size;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.eventlists.EventListInterface #toArray()
	 */
	public synchronized SimEventInterface[] toArray()
	{
		List result = new ArrayList();
		for (Iterator i = this.iterator(); i.hasNext();)
		{
			result.add(i.next());
		}
		return (SimEventInterface[]) result
				.toArray(new SimEventInterface[result.size()]);
	}

	/** ********************** PRIVATE METHODS ******************************** */
	/**
	 * decrements the size
	 */
	private void decrementSize()
	{
		this.modCount++;
		this.size--;
	}

	/**
	 * incrementSize
	 */
	private void incrementSize()
	{
		this.modCount++;
		this.size++;
	}

	/**
	 * returns the first entry from the RedBlackTree
	 * 
	 * @return the first Entry
	 */
	protected Entry firstEntry()
	{
		Entry entry = this.root;
		if (entry != null)
		{
			while (entry.left != null)
			{
				entry = entry.left;
			}
		}
		return entry;
	}

	/**
	 * returns the last entry from the RedBlackTree
	 * 
	 * @return the last Entry
	 */
	private Entry lastEntry()
	{
		Entry entry = this.root;
		if (entry != null)
		{
			while (entry.right != null)
			{
				entry = entry.right;
			}
		}
		return entry;
	}

	/**
	 * searches for nonnull events
	 * 
	 * @param entry the entry to search
	 * @param event the event
	 * @return null?
	 */
	private boolean eventSearchNonNull(final Entry entry,
			final SimEventInterface event)
	{
		if (event.equals(entry.simEvent))
		{
			return true;
		}
		// Check left and right subtrees for event
		return (entry.left != null && eventSearchNonNull(entry.left, event))
				|| (entry.right != null && eventSearchNonNull(entry.right,
						event));
	}

	/**
	 * searches for null events
	 * 
	 * @param entry the entry to search
	 * @return null?
	 */
	private boolean eventSearchNull(final Entry entry)
	{
		if (entry.simEvent == null)
		{
			return true;
		}
		// Check left and right subtrees for event
		return (entry.left != null && eventSearchNull(entry.left))
				|| (entry.right != null && eventSearchNull(entry.right));
	}

	/**
	 * returns the color of an entry
	 * 
	 * @param entry the entry
	 * @return the color
	 */
	private static boolean colorOf(final Entry entry)
	{
		if (entry == null)
		{
			return BLACK;
		}
		return entry.color;
	}

	/**
	 * returns the parent of p
	 * 
	 * @param entry the entry
	 * @return Entry
	 */
	private static Entry parentOf(final Entry entry)
	{
		if (entry == null)
		{
			return null;
		}
		return entry.parent;
	}

	/**
	 * sets the color of p
	 * 
	 * @param entry the entry
	 * @param color the color
	 */
	private static void setColor(final Entry entry, final boolean color)
	{
		if (entry != null)
		{
			entry.color = color;
		}
	}

	/**
	 * returns the left child of entry
	 * 
	 * @param entry the entry
	 * @return Entry
	 */
	private static Entry leftOf(final Entry entry)
	{
		if (entry == null)
		{
			return null;
		}
		return entry.left;
	}

	/**
	 * returns the right entry of entry
	 * 
	 * @param entry the entry
	 * @return Entry
	 */
	private static Entry rightOf(final Entry entry)
	{
		if (entry == null)
		{
			return null;
		}
		return entry.right;
	}

	/**
	 * rotates left
	 * 
	 * @param entry the entry to rotate
	 */
	private void rotateLeft(final Entry entry)
	{
		Entry right = entry.right;
		entry.right = right.left;
		if (right.left != null)
		{
			right.left.parent = entry;
		}
		right.parent = entry.parent;
		if (entry.parent == null)
		{
			this.root = right;
		} else if (entry.parent.left == entry)
		{
			entry.parent.left = right;
		} else
		{
			entry.parent.right = right;
		}
		right.left = entry;
		entry.parent = right;
	}

	/**
	 * rotates to the right
	 * 
	 * @param entry the entry to rotate
	 */
	private void rotateRight(final Entry entry)
	{
		Entry left = entry.left;
		entry.left = left.right;
		if (left.right != null)
		{
			left.right.parent = entry;
		}
		left.parent = entry.parent;
		if (entry.parent == null)
		{
			this.root = left;
		} else if (entry.parent.right == entry)
		{
			entry.parent.right = left;
		} else
		{
			entry.parent.left = left;
		}
		left.right = entry;
		entry.parent = left;
	}

	/**
	 * fixes after insertion
	 * 
	 * @param entry the entry
	 */
	private void fixAfterInsertion(Entry entry)
	{
		entry.color = RED;
		while (entry != null && entry != this.root && entry.parent.color == RED)
		{
			if (parentOf(entry) == leftOf(parentOf(parentOf(entry))))
			{
				Entry y = rightOf(parentOf(parentOf(entry)));
				if (colorOf(y) == RED)
				{
					setColor(parentOf(entry), BLACK);
					setColor(y, BLACK);
					setColor(parentOf(parentOf(entry)), RED);
					entry = parentOf(parentOf(entry));
				} else
				{
					if (entry == rightOf(parentOf(entry)))
					{
						entry = parentOf(entry);
						rotateLeft(entry);
					}
					setColor(parentOf(entry), BLACK);
					setColor(parentOf(parentOf(entry)), RED);
					if (parentOf(parentOf(entry)) != null)
					{
						rotateRight(parentOf(parentOf(entry)));
					}
				}
			} else
			{
				Entry y = leftOf(parentOf(parentOf(entry)));
				if (colorOf(y) == RED)
				{
					setColor(parentOf(entry), BLACK);
					setColor(y, BLACK);
					setColor(parentOf(parentOf(entry)), RED);
					entry = parentOf(parentOf(entry));
				} else
				{
					if (entry == leftOf(parentOf(entry)))
					{
						entry = parentOf(entry);
						rotateRight(entry);
					}
					setColor(parentOf(entry), BLACK);
					setColor(parentOf(parentOf(entry)), RED);
					if (parentOf(parentOf(entry)) != null)
					{
						rotateLeft(parentOf(parentOf(entry)));
					}
				}
			}
		}
		this.root.color = BLACK;
	}

	/**
	 * deletes entry , and then rebalance the tree.
	 * 
	 * @param entry entry
	 */
	protected void deleteEntry(Entry entry)
	{
		this.decrementSize();
		// If strictly internal, copy successor's element to entry and then
		// make entry
		// point to successor.
		if (entry.left != null && entry.right != null)
		{
			Entry s = successor(entry);
			entry.simEvent = s.simEvent;
			entry = s;
		} // entry has 2 children
		// Start fixup at replacement node, if it exists.
		Entry replacement = entry.left;
		if (replacement == null)
		{
			replacement = entry.right;
		}
		if (replacement != null)
		{
			// Link replacement to parent
			replacement.parent = entry.parent;
			if (entry.parent == null)
			{
				this.root = replacement;
			} else if (entry == entry.parent.left)
			{
				entry.parent.left = replacement;
			} else
			{
				entry.parent.right = replacement;
			}
			// Null out links so they are OK to use by fixAfterDeletion.
			entry.left = null;
			entry.right = null;
			entry.parent = null;
			// Fix replacement
			if (entry.color == BLACK)
			{
				fixAfterDeletion(replacement);
			}
		} else if (entry.parent == null)
		{ // return if we are the only node.
			this.root = null;
		} else
		{ //  No children. Use self as phantom replacement and unlink.
			if (entry.color == BLACK)
			{
				fixAfterDeletion(entry);
			}
			if (entry.parent != null)
			{
				if (entry == entry.parent.left)
				{
					entry.parent.left = null;
				} else if (entry == entry.parent.right)
				{
					entry.parent.right = null;
				}
				entry.parent = null;
			}
		}
	}

	/**
	 * fixes after deletion
	 * 
	 * @param entry the entry
	 */
	private void fixAfterDeletion(Entry entry)
	{
		while (entry != this.root && colorOf(entry) == BLACK)
		{
			if (entry == leftOf(parentOf(entry)))
			{
				Entry sib = rightOf(parentOf(entry));
				if (colorOf(sib) == RED)
				{
					setColor(sib, BLACK);
					setColor(parentOf(entry), RED);
					rotateLeft(parentOf(entry));
					sib = rightOf(parentOf(entry));
				}
				if (colorOf(leftOf(sib)) == BLACK
						&& colorOf(rightOf(sib)) == BLACK)
				{
					setColor(sib, RED);
					entry = parentOf(entry);
				} else
				{
					if (colorOf(rightOf(sib)) == BLACK)
					{
						setColor(leftOf(sib), BLACK);
						setColor(sib, RED);
						rotateRight(sib);
						sib = rightOf(parentOf(entry));
					}
					setColor(sib, colorOf(parentOf(entry)));
					setColor(parentOf(entry), BLACK);
					setColor(rightOf(sib), BLACK);
					rotateLeft(parentOf(entry));
					entry = this.root;
				}
			} else
			{ // symmetric
				Entry sib = leftOf(parentOf(entry));
				if (colorOf(sib) == RED)
				{
					setColor(sib, BLACK);
					setColor(parentOf(entry), RED);
					rotateRight(parentOf(entry));
					sib = leftOf(parentOf(entry));
				}
				if (colorOf(rightOf(sib)) == BLACK
						&& colorOf(leftOf(sib)) == BLACK)
				{
					setColor(sib, RED);
					entry = parentOf(entry);
				} else
				{
					if (colorOf(leftOf(sib)) == BLACK)
					{
						setColor(rightOf(sib), BLACK);
						setColor(sib, RED);
						rotateLeft(sib);
						sib = leftOf(parentOf(entry));
					}
					setColor(sib, colorOf(parentOf(entry)));
					setColor(parentOf(entry), BLACK);
					setColor(leftOf(sib), BLACK);
					rotateRight(parentOf(entry));
					entry = this.root;
				}
			}
		}
		setColor(entry, BLACK);
	}

	/**
	 * gets the entry
	 * 
	 * @param event the event
	 * @return Entry the entry
	 */
	private Entry getEntry(final SimEventInterface event)
	{
		Entry entry = this.root;
		while (entry != null)
		{
			int cmp = event.compareTo(entry.simEvent);
			if (cmp < 0)
			{
				entry = entry.left;
			} else if (cmp > 0)
			{
				entry = entry.right;
			} else if (cmp == 0)
			{
				return entry;
			}
		}
		return null;
	}

	/**
	 * Returns the successor of the specified Entry, or null if no such.
	 * 
	 * @param entry the entry
	 * @return Entry the successor
	 */
	protected Entry successor(final Entry entry)
	{
		if (entry == null)
		{
			return null;
		} else if (entry.right != null)
		{
			Entry right = entry.right;
			while (right.left != null)
			{
				right = right.left;
			}
			return right;
		} else
		{
			Entry right = entry.parent;
			Entry ch = entry;
			while (right != null && ch == right.right)
			{
				ch = right;
				right = right.parent;
			}
			return right;
		}
	}

	/**
	 * A Entry <br>
	 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
	 * University of Technology </a>, the Netherlands. <br>
	 * See for project information <a
	 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a>
	 * <br>
	 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
	 * Public License (GPL) </a>, no warranty <br>
	 * 
	 * @version 1.0 Feb 24, 2004 <br>
	 * @author <a
	 *         href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
	 *         Jacobs </a>
	 */
	private static class Entry implements Serializable
	{
		/** the simEvent stored in the entry */
		protected SimEventInterface simEvent = null;

		/** the left entry */
		protected Entry left = null;

		/** the right entry */
		protected Entry right = null;

		/** the parent */
		protected Entry parent = null;

		/** the color of the entry */
		protected boolean color = BLACK;

		/**
		 * constructs a new Entry
		 */
		public Entry()
		{
			super();
		}

		/**
		 * constructs a new Entry
		 * 
		 * @param simEvent the simEvent of the entry
		 */
		public Entry(final SimEventInterface simEvent)
		{
			this();
			this.simEvent = simEvent;
		}

		/**
		 * constructs a new Entry
		 * 
		 * @param simEvent the simEvent of the entry
		 * @param parent the parent
		 */
		public Entry(final SimEventInterface simEvent, final Entry parent)
		{
			this(simEvent);
			this.parent = parent;
		}

		/**
		 * constructs a new Entry
		 * 
		 * @param simEvent the simEvent of the entry
		 * @param parent the parent
		 * @param left the left child
		 * @param right the right child
		 */
		public Entry(final SimEventInterface simEvent, final Entry parent,
				final Entry left, final Entry right)
		{
			this(simEvent, parent);
			this.left = left;
			this.right = right;
		}
	}
	/**
	 * A EntryIterator <br>
	 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
	 * University of Technology </a>, the Netherlands. <br>
	 * See for project information <a
	 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a>
	 * <br>
	 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
	 * Public License (GPL) </a>, no warranty <br>
	 * 
	 * @version 1.0 Feb 16, 2004 <br>
	 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">
	 *         Peter Jacobs </a>
	 */
	private class EntryIterator implements Iterator
	{
		/** the expectedModCount */
		protected int expectedModCount = RedBlackTree.this.modCount;

		/** the event of the lastReturned entry */
		protected Entry lastReturned = null;

		/** the event of the next entry */
		protected Entry next = null;

		/**
		 * constructs a new EntryIterator
		 */
		public EntryIterator()
		{
			this.next = firstEntry();
		}

		/**
		 * constructs a new EntryIterator
		 * 
		 * @param first the starting point
		 */
		public EntryIterator(final Entry first)
		{
			this.next = first;
		}

		/**
		 * @see java.util.Iterator#hasNext()
		 */
		public boolean hasNext()
		{
			return this.next != null;
		}

		/**
		 * @return the next Entry
		 */
		public final Entry nextEntry()
		{
			if (this.next == null)
			{
				throw new NoSuchElementException();
			}
			if (RedBlackTree.this.modCount != this.expectedModCount)
			{
				throw new ConcurrentModificationException();
			}
			this.lastReturned = this.next;
			this.next = successor(this.next);
			return this.lastReturned;
		}

		/**
		 * @see java.util.Iterator#next()
		 */
		public Object next()
		{
			return this.nextEntry();
		}

		/**
		 * @see java.util.Iterator#remove()
		 */
		public void remove()
		{
			if (this.lastReturned == null)
			{
				throw new IllegalStateException();
			}
			if (RedBlackTree.this.modCount != this.expectedModCount)
			{
				throw new ConcurrentModificationException();
			}
			if (this.lastReturned.left != null
					&& this.lastReturned.right != null)
			{
				this.next = this.lastReturned;
			}
			deleteEntry(this.lastReturned);
			this.expectedModCount++;
			this.lastReturned = null;
		}
	}
	/**
	 * A EventListIterator <br>
	 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
	 * University of Technology </a>, the Netherlands. <br>
	 * See for project information <a
	 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a>
	 * <br>
	 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
	 * Public License (GPL) </a>, no warranty <br>
	 * 
	 * @version 1.0 Feb 16, 2004 <br>
	 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">
	 *         Peter Jacobs </a>
	 */
	private class EventListIterator extends EntryIterator
	{
		/**
		 * constructs a new EventListIterator
		 */
		public EventListIterator()
		{
			super();
		}

		/**
		 * @see java.util.Iterator#next()
		 */
		public Object next()
		{
			return ((Entry) super.next()).simEvent;
		}
	}
}