/*
 * @(#)Experiment.java Aug 18, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.experiment;

import java.io.Serializable;
import java.net.URL;
import java.text.DateFormat;
import java.util.Calendar;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import nl.tudelft.simulation.dsol.ModelInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.event.EventProducerInterface;
import nl.tudelft.simulation.event.EventType;
import nl.tudelft.simulation.logger.Logger;

/**
 * The Experiment specifies the parameters for a simulation experiment <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>, <a
 *         href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 *         Verbraeck </a>
 */
public class Experiment extends EventProducer implements
		EventListenerInterface, Serializable
{
	/** END_OF_EXPERIMENT_EVENT is fired when the experiment is ended */
	public static final EventType END_OF_EXPERIMENT_EVENT = new EventType(
			"END_OF_EXPERIMENT_EVENT");

	/** MODEL_CHANGED_EVENT is fired whenever the model is changed */
	public static final EventType MODEL_CHANGED_EVENT = new EventType(
			"MODEL_CHANGED_EVENT");

	/** SIMULATOR_CHANGED_EVENT is fired whenever the simulator is changed */
	public static final EventType SIMULATOR_CHANGED_EVENT = new EventType(
			"SIMULATOR_CHANGED_EVENT");

	/** EXPERIMENT_NAME the name property */
	public static final String EXPERIMENT_NAME = "EXPERIMENT_NAME";

	/** EXPERIMENT_ANALYST the analyst name property */
	public static final String EXPERIMENT_ANALYST = "EXPERIMENT_ANALYST";

	/**
	 * treatments represent the treatments of this experiment
	 * 
	 * @uml.property name="treatments"
	 */
	private Treatment[] treatments = null;


	/** properties reflect the properties */
	private Properties properties = new Properties();

	/** treatment reflects the current treatment */
	private int treatment = 0;

	/**
	 * simulator reflects the simulator
	 * 
	 * @uml.property name="simulator"
	 */
	private SimulatorInterface simulator;

	/**
	 * model reflects the model
	 * 
	 * @uml.property name="model"
	 */
	private ModelInterface model;

	/**
	 * the current run
	 * 
	 * @uml.property name="run"
	 */
	private String run;

	/**
	 * the URL of the Experiment
	 * 
	 * @uml.property name="url"
	 */
	private URL url = null;


	/**
	 * constructs a new Experiment
	 */
	public Experiment()
	{
		super();
	}

	/**
	 * sets the simulator
	 * 
	 * @param simulator the simulator
	 * 
	 * @uml.property name="simulator"
	 */
	public synchronized void setSimulator(final SimulatorInterface simulator)
	{
		this.simulator = simulator;
		this.fireEvent(SIMULATOR_CHANGED_EVENT, simulator);
	}

	/**
	 * returns the simulator
	 * 
	 * @return SimulatorInterface
	 * 
	 * @uml.property name="simulator"
	 */
	public SimulatorInterface getSimulator()
	{
		return this.simulator;
	}

	/**
	 * returns the current run
	 * 
	 * @return String
	 * 
	 * @uml.property name="run"
	 */
	public String getRun()
	{
		return this.run;
	}


	/**
	 * starts the experiment on a simulator
	 */
	public synchronized void start()
	{
		Calendar calendar = Calendar.getInstance();
		calendar.setTimeInMillis(System.currentTimeMillis());
		this.run = DateFormat.getDateTimeInstance().format(calendar.getTime());
		try
		{
			new InitialContext().createSubcontext(this.run);
		} catch (NamingException exception)
		{
			Logger.warning(this, "start", exception);
		}
		this.treatment = 0;
		this.notify(new Event(RunControl.END_OF_RUN_EVENT, this, null));
	}

	/**
	 * returns a property
	 * 
	 * @param key properties are stored as key-value
	 * @return String the property
	 */
	public String getProperty(final String key)
	{
		return this.properties.getProperty(key);
	}

	/**
	 * sets a property
	 * 
	 * @param key properties are stored in key-value pairs
	 * @param value properties are stored in key-value pairs
	 */
	public void setProperty(final String key, final String value)
	{
		this.properties.put(key, value);
	}

	/**
	 * returns the model
	 * 
	 * @return ModelInterface the model
	 * 
	 * @uml.property name="model"
	 */
	public ModelInterface getModel()
	{
		return this.model;
	}

	/**
	 * returns the URL
	 * 
	 * @return URL
	 * 
	 * @uml.property name="url"
	 */
	public URL getUrl()
	{
		return this.url;
	}


	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event)
	{
		try
		{
			Context context = (Context) new InitialContext().lookup(this.run);
			if (event.getType().equals(RunControl.END_OF_RUN_EVENT))
			{
				if (this.treatment < this.treatments.length)
				{
					context.createSubcontext("treatment(" + this.treatment
							+ ")");
					this.treatments[this.treatment]
							.getRunControl()
							.addListener(this, RunControl.END_OF_RUN_EVENT,
									EventProducerInterface.LAST_POSITION, false);
					this.treatment++;
					this.treatments[this.treatment - 1].getRunControl().start(
							this.simulator);
				} else
				{
					this.fireEvent(new Event(
							Experiment.END_OF_EXPERIMENT_EVENT, this, null));
				}
			}
		} catch (Exception exception)
		{
			Logger.warning(this, "notify", exception);
		}
	}

	/**
	 * resets the experiment
	 */
	public synchronized void reset()
	{
		this.treatment = 0;
		for (int i = 0; i < this.treatments.length; i++)
		{
			this.treatments[i].getRunControl().reset();
		}
		try
		{
			this.setSimulator((SimulatorInterface) this.simulator.getClass()
					.newInstance());
		} catch (Exception exception)
		{
			exception.printStackTrace();
		}
	}

	/**
	 * sets the model on the experiment
	 * 
	 * @param model the simulatormodel
	 * 
	 * @uml.property name="model"
	 */
	public synchronized void setModel(final ModelInterface model)
	{
		this.model = model;
		this.fireEvent(MODEL_CHANGED_EVENT, model);
	}

	/**
	 * returns the treatments
	 * 
	 * @return Treatment
	 * 
	 * @uml.property name="treatments"
	 */
	public Treatment[] getTreatments()
	{
		return this.treatments;
	}

	/**
	 * sets the treatments
	 * 
	 * @param treatments reflect the treatments
	 * 
	 * @uml.property name="treatments"
	 */
	public void setTreatments(final Treatment[] treatments)
	{
		this.treatments = treatments;
	}

	/**
	 * sets the URL of the experiment
	 * 
	 * @param url the URL
	 * 
	 * @uml.property name="url"
	 */
	public void setUrl(final URL url)
	{
		this.url = url;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		String result = "[" + super.toString() + " ; "
				+ this.getProperty("dsol.experiment.projectTitle") + " ; "
				+ this.getProperty("dsol.experiment.analystName")
				+ " ; treatments=";
		for (int i = 0; i < this.treatments.length; i++)
		{
			result = result + this.treatments[i].toString() + " ; ";
		}
		result = result.substring(0, result.length() - 2) + "]";
		return result;
	}
}