/*
 * @(#)Replication.java Aug 18, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.experiment;

import java.io.Serializable;
import java.rmi.RemoteException;

import javax.naming.Context;
import javax.naming.InitialContext;

import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.event.EventProducerInterface;
import nl.tudelft.simulation.event.EventType;
import nl.tudelft.simulation.logger.Logger;

/**
 * The RunControl specifies all parameters for the simulator. <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>, <a
 *         href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 *         Verbraeck </a>
 */
public class RunControl extends EventProducer implements Serializable,
		EventListenerInterface
{
	/** END_OF_RUN_EVENT is fired at the end of a run */
	public static final EventType END_OF_RUN_EVENT = new EventType(
			"END_OF_RUN_EVENT");

	/**
	 * treatment reflects the treatment to which this runcontrol belongs
	 * 
	 * @uml.property name="treatment"
	 */
	protected Treatment treatment = null;

	/**
	 * replications are the replications of this runControl
	 * 
	 * @uml.property name="replications"
	 */
	private Replication[] replications;

	/**
	 * warmupPeriod is the warmup period
	 * 
	 * @uml.property name="warmupPeriod"
	 */
	private double warmupPeriod = 0.0;

	/**
	 * runLength reflects the runLength of the runControl
	 * 
	 * @uml.property name="runLength"
	 */
	private double runLength = Double.NaN;


	/** replication refers to the current replication */
	private int replication = 0;

	/**
	 * Constructs a new RunControl object.
	 * 
	 * @param treatment The parent treatment
	 */
	public RunControl(final Treatment treatment)
	{
		super();
		this.treatment = treatment;
		this.addListener(treatment.getRunControl(),
				RunControl.END_OF_RUN_EVENT, false);
	}

	/**
	 * starts the runControl
	 * 
	 * @param simulator is the simulator
	 * @throws RemoteException on network failure
	 */
	public void start(final SimulatorInterface simulator)
			throws RemoteException
	{
		//Let's run the runControl by faking the first
		//END_OF_REPLICATION_EVENT
		simulator.addListener(this,
				SimulatorInterface.END_OF_REPLICATION_EVENT,
				EventProducerInterface.LAST_POSITION, false);
		this.replication = 0;
		this.notify(new Event(SimulatorInterface.END_OF_REPLICATION_EVENT,
				simulator, null));
	}

	/**
	 * returns the treatment
	 * 
	 * @return treatment
	 * 
	 * @uml.property name="treatment"
	 */
	public Treatment getTreatment()
	{
		return this.treatment;
	}

	/**
	 * returns the replications defined in this runControl
	 * 
	 * @return Replication the replications
	 * 
	 * @uml.property name="replications"
	 */
	public Replication[] getReplications()
	{
		return this.replications;
	}

	/**
	 * returns the runLength
	 * 
	 * @return double the runlength
	 * 
	 * @uml.property name="runLength"
	 */
	public double getRunLength()
	{
		return this.runLength;
	}

	/**
	 * returns the warmupPeriod
	 * 
	 * @return double the warmup period
	 * 
	 * @uml.property name="warmupPeriod"
	 */
	public double getWarmupPeriod()
	{
		return this.warmupPeriod;
	}


	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event)
	{
		try
		{
			if (event.getType().equals(
					SimulatorInterface.END_OF_REPLICATION_EVENT))
			{
				SimulatorInterface simulator = (SimulatorInterface) event
						.getSource();
				if (this.replication < this.replications.length)
				{
					Context context = (Context) new InitialContext()
							.lookup(this.treatment.getExperiment().getRun()
									+ "/treatment("
									+ this.treatment.getNumber() + ")");
					context.createSubcontext("replication(" + this.replication
							+ ")");
					context.createSubcontext("replication(" + this.replication
							+ ")/animation");
					context.createSubcontext("replication(" + this.replication
							+ ")/animation/2D");
					context.createSubcontext("replication(" + this.replication
							+ ")/animation/3D");
					simulator.initialize(this.replications[this.replication]);
					this.replication++;
					simulator.start();
				} else
				{
					this.fireEvent(new Event(RunControl.END_OF_RUN_EVENT, this,
							null));
				}
			}
		} catch (Exception exception)
		{
			Logger.warning(this, "notify", exception);
		}
	}

	/**
	 * resets the RunControl
	 */
	public synchronized void reset()
	{
		this.replication = 0;
		for (int i = 0; i < this.replications.length; i++)
		{
			this.replications[i].reset();
		}
	}

	/**
	 * sets the replications
	 * 
	 * @param replications the replications of the runControl
	 * 
	 * @uml.property name="replications"
	 */
	public void setReplications(final Replication[] replications)
	{
		this.replications = replications;
	}

	/**
	 * sets the runLength
	 * 
	 * @param runLength which is the runLength
	 * 
	 * @uml.property name="runLength"
	 */
	public void setRunLength(final double runLength)
	{
		this.runLength = runLength;
	}

	/**
	 * sets the warmupPeriod
	 * 
	 * @param warmupPeriod reflects the warmupPeriod
	 * 
	 * @uml.property name="warmupPeriod"
	 */
	public void setWarmupPeriod(final double warmupPeriod)
	{
		this.warmupPeriod = warmupPeriod;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		String result = super.toString() + " ; RL=" + this.getRunLength()
				+ " ; WP=" + this.getWarmupPeriod() + " ; replications=[";
		for (int i = 0; i < this.replications.length; i++)
		{
			result = result + this.replications[i].toString() + " ; ";
		}
		result = result.substring(0, result.length() - 2) + "]";
		return "[" + result;
	}
}