/*
 * @(#)Resource.java Feb 1, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.formalisms;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.event.EventType;

/**
 * A resource defines a shared and limited amount.
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.2 2004-03-26
 * @since 1.0
 */
public class Resource extends EventProducer
{
	/** UTILIZATION_EVENT is fired on activity */
	public static final EventType UTILIZATION_EVENT = new EventType(
			"UTILIZATION_EVENT");

	/** RESOURCE_REQUESTED_QUEUE_LENGTH fired on changes in queue length */
	public static final EventType RESOURCE_REQUESTED_QUEUE_LENGTH = new EventType(
			"RESOURCE_REQUESTED_QUEUE_LENGTH");

	/**
	 * capacity defines the maximuum capacity of the resource
	 * 
	 * @uml.property name="capacity"
	 */
	protected double capacity;


	/** claimedCapacity defines the currently claimed capacity */
	protected double claimedCapacity = 0.0;

	/** request defines the list of requestors for this resource */
	private List requests = Collections.synchronizedList(new ArrayList());

	/** simulator defines the simulator on which is scheduled */
	protected DEVSSimulatorInterface simulator;

	/** the description of the resource */
	protected String description = "resource";

	/**
	 * Method Resource.
	 * 
	 * @param simulator on which is scheduled
	 * @param description the description of this resource
	 * @param capacity of the resource
	 */
	public Resource(final DEVSSimulatorInterface simulator,
			final String description, final double capacity)
	{
		super();
		this.description = description;
		this.simulator = simulator;
		this.capacity = capacity;
	}

	/**
	 * Method Resource.
	 * 
	 * @param simulator on which is scheduled
	 * @param capacity of the resource
	 */
	public Resource(final DEVSSimulatorInterface simulator,
			final double capacity)
	{
		this(simulator, "resource", capacity);
	}

	/**
	 * returns the capacity of the resource
	 * 
	 * @return capacity
	 * 
	 * @uml.property name="capacity"
	 */
	public double getCapacity()
	{
		return this.capacity;
	}


	/**
	 * Method alterClaimedCapacity.
	 * 
	 * @param amount refers the amount which is added to the claimed capacity
	 * @throws RemoteException on network failure
	 */
	private synchronized void alterClaimedCapacity(final double amount)
			throws RemoteException
	{
		this.claimedCapacity += amount;
		this.fireEvent(Resource.UTILIZATION_EVENT, this.claimedCapacity,
				this.simulator.getSimulatorTime());
	}

	/**
	 * sets the capacity of the resource
	 * 
	 * @param capacity the new maximal capacity
	 * 
	 * @uml.property name="capacity"
	 */
	public void setCapacity(final double capacity)
	{
		this.capacity = capacity;
	}

	/**
	 * requests an amount of capacity from the resource \
	 * 
	 * @param amount the requested amount
	 * @param requestor the RequestorInterface requesting the amount
	 * @throws RemoteException on network failure
	 * @throws SimRuntimeException on other failures
	 */
	public synchronized void requestCapacity(final double amount,
			final ResourceRequestorInterface requestor) throws RemoteException,
			SimRuntimeException
	{
		if (amount < 0.0)
		{
			throw new SimRuntimeException(
					"requested capacity on resource cannot <0.0");
		}
		if ((this.claimedCapacity + amount) <= this.capacity)
		{
			this.alterClaimedCapacity(amount);
			this.simulator.scheduleEvent(0, this, requestor,
					"receiveRequestedResource", new Object[]{
							new Double(amount), this});
		} else
		{
			synchronized (this.requests)
			{
				this.requests.add(new Request(requestor, amount));
			}
			this.fireEvent(Resource.RESOURCE_REQUESTED_QUEUE_LENGTH,
					(double) this.requests.size(), this.simulator
							.getSimulatorTime());
		}
	}

	/**
	 * releases an amount of capacity from the resource.
	 * 
	 * @param amount the amount to release
	 * @throws RemoteException on network failure
	 */
	public void releaseCapacity(final double amount) throws RemoteException
	{
		if (amount < 0.0)
		{
			throw new IllegalArgumentException(
					"released capacity on resource cannot <0.0");
		}
		this.alterClaimedCapacity(-amount);
		synchronized (this.requests)
		{
			for (Iterator i = this.requests.iterator(); i.hasNext();)
			{
				Request request = (Request) i.next();
				if ((this.capacity - this.claimedCapacity) >= request
						.getAmount())
				{
					this.alterClaimedCapacity(amount);
					request.getRequestor().receiveRequestedResource(
							request.getAmount(), this);
					synchronized (this.requests)
					{
						this.requests.remove(request);
					}
					this.fireEvent(Resource.RESOURCE_REQUESTED_QUEUE_LENGTH,
							(double) this.requests.size(), this.simulator
									.getSimulatorTime());
					if ((this.capacity - this.claimedCapacity) >= 0.0)
					{
						this.releaseCapacity(0);
					}
					return;
				}
			}
		}
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return this.description;
	}

	/**
	 * A Request.
	 */
	public class Request
	{
		/** requestor the resourceRequestor */
		private ResourceRequestorInterface requestor;

		/** amount is the amount requested by the resource */
		private double amount;

		/**
		 * constructs a new Request
		 * 
		 * @param requestor the requestor
		 * @param amount the requested amount
		 */
		public Request(final ResourceRequestorInterface requestor,
				final double amount)
		{
			this.requestor = requestor;
			this.amount = amount;
		}

		/**
		 * gets the requested amount.
		 * 
		 * @return the requested amount
		 */
		public double getAmount()
		{
			return this.amount;
		}

		/**
		 * gets the requestor.
		 * 
		 * @return the Requestor.
		 */
		public ResourceRequestorInterface getRequestor()
		{
			return this.requestor;
		}
	}
}