/*
 * @(#) DifferentialEquation.java Oct 25, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.formalisms.dess;

import java.rmi.RemoteException;

import nl.tudelft.simulation.dsol.simulators.DESSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.event.EventType;
import nl.tudelft.simulation.jstats.ode.integrators.NumericalIntegrator;
import nl.tudelft.simulation.logger.Logger;

/**
 * The Differential equation provides a reference implementation of the
 * differential equation.
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.4 2004-03-26
 * @since 1.0
 */
public abstract class DifferentialEquation extends
		nl.tudelft.simulation.jstats.ode.DifferentialEquation implements
		DifferentialEquationInterface, EventListenerInterface
{
	//we initialze the array (30 size set seems enough..)
	static
	{
		for (int i = 0; i < VALUE_CHANGED_EVENT.length; i++)
		{
			VALUE_CHANGED_EVENT[i] = new EventType("VALUE_CHANGED_EVENT[" + i
					+ "]");
		}
	}

	/** simulator */
	protected DESSSimulatorInterface simulator = null;

	/** the previousX */
	protected double previousX = 0.0;

	/** the previousY */
	protected double[] previousY = null;

	/**
	 * constructs a new stateful DifferentialEquation with Euleras numerical
	 * integration method.
	 * 
	 * @param simulator the simulator
	 * @throws RemoteException on network exception
	 */
	public DifferentialEquation(final DESSSimulatorInterface simulator)
			throws RemoteException
	{
		this(simulator, simulator.getTimeStep(),
				NumericalIntegrator.DEFAULT_INTEGRATOR);
	}

	/**
	 * constructs a new stateful DifferentialEquation with Euleras numerical
	 * integration method.
	 * 
	 * @param simulator the simulator
	 * @param timeStep the timeStep for ODE estimation
	 */
	public DifferentialEquation(final DESSSimulatorInterface simulator,
			final double timeStep)
	{
		this(simulator, timeStep, NumericalIntegrator.DEFAULT_INTEGRATOR);
	}

	/**
	 * constructs a new DifferentialEquation
	 * 
	 * @param simulator the simulator.
	 * @param timeStep the timeStep for ODE estimation.
	 * @param numericalMethod the numerical method to be used.
	 */
	public DifferentialEquation(final DESSSimulatorInterface simulator,
			final double timeStep, final short numericalMethod)
	{
		super(timeStep, numericalMethod);
		this.simulator = simulator;
		try
		{
			simulator.addListener(this, SimulatorInterface.TIME_CHANGED_EVENT,
					false);
		} catch (RemoteException exception)
		{
			Logger.warning(this, "DifferentialEquation", exception);
		}
	}

	/**
	 * constructs a new DifferentialEquation
	 * 
	 * @param simulator the simulator.
	 * @param timeStep the timeStep for ODE estimation.
	 * @param numericalIntegrator the actual integrator to be used.
	 */
	public DifferentialEquation(final DESSSimulatorInterface simulator,
			double timeStep, final NumericalIntegrator numericalIntegrator)
	{
		super(timeStep, numericalIntegrator);
		this.simulator = simulator;
		try
		{
			simulator.addListener(this, SimulatorInterface.TIME_CHANGED_EVENT,
					false);
		} catch (RemoteException exception)
		{
			Logger.warning(this, "DifferentialEquation", exception);
		}
	}

	/**
	 * @see EventListenerInterface#notify(EventInterface)
	 */
	public synchronized void notify(final EventInterface event)
			throws RemoteException
	{
		if (event.getSource() instanceof DESSSimulatorInterface
				&& event.getType()
						.equals(SimulatorInterface.TIME_CHANGED_EVENT))
		{
			if (this.simulator.getSimulatorTime() < super.x0
					|| Double.isNaN(super.x0))
			{
				return;
			}
			// do not put super here!
			this.previousY = integrateY(this.simulator.getSimulatorTime(),
					this.previousX, this.previousY);
			for (int i = 0; i < super.y0.length; i++)
			{
				this.fireEvent(
						DifferentialEquationInterface.VALUE_CHANGED_EVENT[i],
						this.previousY[i], this.simulator.getSimulatorTime());
			}
			this.previousX = this.simulator.getSimulatorTime();
		}
	}

	/**
	 * @see nl.tudelft.simulation.jstats.ode.DifferentialEquationInterface#initialize(double,
	 *      double[])
	 */
	public void initialize(double x, double[] y)
	{
		super.initialize(x, y);
		this.previousX = x;
		this.previousY = y;
	}
}