/*
 * @(#)SimEvent.java Feb 1, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.formalisms.devs;

import java.io.Serializable;

import nl.tudelft.simulation.dsol.SimRuntimeException;

/**
 * The AbstractSimEvent forms the basement for SimEvents and defines a compare
 * method by which eventLists can compare priority of the event.
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.11 2004-03-26
 * @since 1.0
 */
public abstract class AbstractSimEvent implements SimEventInterface,
		Comparable, Serializable
{

	/**
	 * absoluteExecutionTime reflects the time at which the event is scheduled
	 * 
	 * @uml.property name="absoluteExecutionTime"
	 */
	protected double absoluteExecutionTime = Double.NaN;

	/**
	 * priority reflects the priority of the event
	 * 
	 * @uml.property name="priority"
	 */
	protected short priority = SimEventInterface.NORMAL_PRIORITY;


	/** the id used in compare statements */
	protected long id = 0L;

	/**
	 * The constuctor of the event stores the time the event must be executed
	 * and the object and method to invoke
	 * 
	 * @param executionTime reflects the time the event has to be executed.
	 */
	public AbstractSimEvent(final double executionTime)
	{
		this(executionTime, SimEventInterface.NORMAL_PRIORITY);
	}

	/**
	 * The constuctor of the event stores the time the event must be executed
	 * and the object and method to invoke
	 * 
	 * @param executionTime reflects the time the event has to be executed.
	 * @param priority reflects the priority of the event
	 */
	public AbstractSimEvent(final double executionTime, final short priority)
	{
		this.absoluteExecutionTime = executionTime;
		if (priority < SimEventInterface.MIN_PRIORITY - 1
				|| priority > SimEventInterface.MAX_PRIORITY + 1)
		{
			throw new IllegalArgumentException("priority must be between ["
					+ SimEventInterface.MIN_PRIORITY + ".."
					+ SimEventInterface.MAX_PRIORITY + "]");
		}
		this.priority = priority;
	}

	/**
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(final Object object)
	{
		SimEventInterface simEvent = (SimEventInterface) object;
		if (this.absoluteExecutionTime < simEvent.getAbsoluteExecutionTime())
		{
			return -1;
		}
		if (this.absoluteExecutionTime > simEvent.getAbsoluteExecutionTime())
		{
			return 1;
		}
		if (this.priority < simEvent.getPriority())
		{
			return 1;
		}
		if (this.priority > simEvent.getPriority())
		{
			return -1;
		}
		if (this.id < simEvent.getID())
		{
			return -1;
		}
		if (this.id > simEvent.getID())
		{
			return 1;
		}
		return 0;
	}

	/**
	 * executes the simEvent
	 * 
	 * @throws SimRuntimeException on execution failure
	 */
	public abstract void execute() throws SimRuntimeException;

	/**
	 * @return The execution time of a simulation event
	 * 
	 * @uml.property name="absoluteExecutionTime"
	 */
	public double getAbsoluteExecutionTime()
	{
		return this.absoluteExecutionTime;
	}

	/**
	 * @return The priority of a simulation event. The priorities are programmed
	 *         according to the Java thread priority. Use 10 (MAX_PRIORITY), -9, .. ,
	 *         5 (NORMAL_PRIORITY), 1(MIN_PRIORITY)
	 * 
	 * @uml.property name="priority"
	 */
	public short getPriority()
	{
		return this.priority;
	}

	/**
	 * @return Returns the id.
	 */
	public long getID()
	{
		return this.id;
	}

	/**
	 * @param id The id to set.
	 */
	public void setID(final long id)
	{
		this.id = id;
	}
}