package nl.tudelft.simulation.dsol.formalisms.flow;

import java.util.Collections;
import java.util.SortedMap;
import java.util.TreeMap;

import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.formalisms.devs.SimEvent;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.jstats.distributions.DistContinuous;
import nl.tudelft.simulation.logger.Logger;

/**
 * The schedule is an extension to the generate which accepts a schedule of
 * interarrival times. Instead of generating with a continuous interarrival
 * distribution we submit a map consiting of keys (execution times). Each key
 * indicates the <i>starting time </i> of a new interval, while the value in the
 * map is the continuous distribution function to use to draw the interarrival
 * times. If no values have to be generated in a certain interval, use a large
 * interarrival time value in the distribution function, or use
 * DistConstant(stream, 1E20) to indicate that the next drawing will take place
 * <i>after </i> the end of the interval. <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>, <a
 *         href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 *         Verbraeck </a>
 */
public class Schedule extends Generator
{
	/** schedule is a time sorted map of distributions */
	private SortedMap schedule = Collections
			.synchronizedSortedMap(new TreeMap());

	/**
	 * constructs a new Schedule
	 * 
	 * @param simulator is the on which the construction of the objects must be
	 *        scheduled.
	 * @param myClass is the class of which entities are created
	 * @param constructorArguments are the parameters for the constructor of
	 *        myClass. of arguments.
	 *        <code>constructorArgument[n]=new Integer(12)</code> may have
	 *        constructorArgumentClasses[n]=int.class;
	 * @throws SimRuntimeException on constructor invokation.
	 */
	public Schedule(final DEVSSimulatorInterface simulator,
			final Class myClass, final Object[] constructorArguments)
			throws SimRuntimeException
	{
		super(simulator, myClass, constructorArguments);
	}

	/**
	 * returns the schedule
	 * 
	 * @return SortedMap the schedule
	 */
	public SortedMap getSchedule()
	{
		return this.schedule;
	}

	/**
	 * sets the schedule
	 * 
	 * @param map is the new map
	 */
	public synchronized void setSchedule(final SortedMap map)
	{
		this.schedule = map;
		this.schedule.putAll(map);
		this.changeIntervalTime();
	}

	/**
	 * changes the intervalTime of the schedule
	 */
	public synchronized void changeIntervalTime()
	{
		try
		{
			if (!this.schedule.isEmpty())
			{
				this.simulator.cancelEvent(super.nextEvent);
				this.interval = (DistContinuous) this.schedule.values()
						.iterator().next();
				this.schedule.remove(this.schedule.firstKey());
				this.simulator.scheduleEvent(new SimEvent(
						((Double) this.schedule.firstKey()).doubleValue(),
						this, this, "changeIntervalTime", null));
				this.generate(this.constructorArguments);
				Logger.finest(this, "changeIntervalTime",
						"set the intervalTime to " + this.interval);
			}
		} catch (Exception exception)
		{
			Logger.warning(this, "changeIntervalTime", exception);
		}
	}
}