/*
 * @(#) Process.java Jan 19, 2004
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.formalisms.process;

import java.rmi.RemoteException;
import java.util.Stack;

import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.formalisms.devs.SimEvent;
import nl.tudelft.simulation.dsol.interpreter.Frame;
import nl.tudelft.simulation.dsol.interpreter.Interpreter;
import nl.tudelft.simulation.dsol.interpreter.InterpreterException;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.language.reflection.ClassUtil;
import nl.tudelft.simulation.logger.Logger;

/**
 * A Process <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 1.0 Jan 19, 2004 <br>
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 */
public abstract class Process
{
	/** the processStack of this process */
	private Stack frameStack = new Stack();

	/** the simulator to schedule on */
	protected DEVSSimulatorInterface simulator = null;

	/**
	 * constructs a new Process
	 * 
	 * @param simulator the simulator to schedule on
	 */
	public Process(final DEVSSimulatorInterface simulator)
	{
		super();
		this.simulator = simulator;
		try
		{
			this.frameStack.push(Interpreter.createFrame(this, ClassUtil
					.resolveMethod(this, "process", null), null));
			double simulatorTime = this.simulator.getSimulatorTime();
			if (Double.isNaN(simulatorTime))
			{
				simulatorTime = 0.0;
			}
			SimEvent simEvent = new SimEvent(simulatorTime, this, this,
					"resume", null);
			this.simulator.scheduleEvent(simEvent);
		} catch (Exception exception)
		{
			exception.printStackTrace();
			Logger.severe(this, "<init>", exception);
		}
	}

	/**
	 * processes the process.
	 * 
	 * @throws SimRuntimeException on simulation failure
	 * @throws RemoteException on remote network failure
	 */
	public abstract void process() throws SimRuntimeException, RemoteException;

	/**
	 * holds the process for a duration
	 * 
	 * @param duration the duration
	 * @throws SimRuntimeException on negative duration
	 * @throws RemoteException on network failure
	 */
	protected void hold(final double duration) throws SimRuntimeException,
			RemoteException
	{
		//First we schedule the resume operation
		this.simulator.scheduleEvent(new SimEvent(this.simulator
				.getSimulatorTime()
				+ duration, this, this, "resume", null));

		//Now we suspend
		this.suspend();
	}

	/**
	 * resumes this process
	 */
	public synchronized void resume()
	{
		if (this.frameStack.isEmpty())
		{
			return;
		}
		try
		{
			((Frame) this.frameStack.peek()).setPaused(false);
			Object result = Interpreter.interpret(this.frameStack);
			if (result instanceof Stack)
			{
				this.frameStack = (Stack) result;
			}
		} catch (InterpreterException exception)
		{
			Logger.warning(this, "resume", exception);
		}
	}

	/**
	 * suspends the process. This method throughs an exception whenever it is
	 * not interpreted but executed directly on the JVM.
	 * 
	 * @throws SimRuntimeException on direct invokation.
	 */
	protected void suspend() throws SimRuntimeException
	{
		throw new SimRuntimeException("suspend should be interpreted."
				+ " One may not invoke this method directly");
	}
}