/*
 * @(#)DESSSimulator.java Aug 18, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.simulators;

import java.rmi.RemoteException;

import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.experiment.Replication;
import nl.tudelft.simulation.logger.Logger;

/**
 * The DESS defines the interface of the DESS simulator. DESS stands for the
 * Differential Equation System Specification. More information on Modeling &
 * Simulation can be found in "Theory of Modeling and Simulation" by Bernard
 * Zeigler et. al. <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>, <a
 *         href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 *         Verbraeck </a>
 */
public class DESSSimulator extends Simulator implements DESSSimulatorInterface
{

	/**
	 * timeStep represents the timestep of the DESS simulator
	 * 
	 * @uml.property name="timeStep"
	 */
	protected double timeStep = DEFAULT_TIME_STEP;


	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface
	 *      #initialize(nl.tudelft.simulation.dsol.experiment.Replication)
	 */
	public void initialize(final Replication replication)
			throws RemoteException, SimRuntimeException
	{
		super.initialize(replication);
		this.replication.getRunControl().getTreatment().getExperiment()
				.getModel().constructModel(this);
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.DESSSimulatorInterface
	 *      #getTimeStep()
	 * 
	 * @uml.property name="timeStep"
	 */
	public double getTimeStep()
	{
		return this.timeStep;
	}


	/**
	 * @see nl.tudelft.simulation.dsol.simulators.Simulator#run()
	 */
	public void run()
	{
		while (this.simulatorTime <= this.replication.getRunControl()
				.getRunLength()
				&& isRunning())
		{
			synchronized (super.semaphore)
			{
				this.simulatorTime = this.simulatorTime + this.timeStep;
				if (this.simulatorTime > this.replication.getRunControl()
						.getRunLength())
				{
					this.simulatorTime = this.replication.getRunControl()
							.getRunLength();
					this.stop();
				}
				this.fireEvent(SimulatorInterface.TIME_CHANGED_EVENT,
						this.simulatorTime, this.simulatorTime);
			}
		}
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.DESSSimulatorInterface
	 *      #setTimeStep(double)
	 * 
	 * @uml.property name="timeStep"
	 */
	public void setTimeStep(final double timeStep)
	{
		synchronized (super.semaphore)
		{
			if (timeStep < 0)
			{
				throw new IllegalArgumentException("timeStep <0 ?");
			}
			this.timeStep = timeStep;
			Logger
					.finer(this, "setTimeStep", "set the timeStep to "
							+ timeStep);
			this.fireEvent(TIME_STEP_CHANGED_EVENT, timeStep);
		}
	}

}