/*
 * @(#)Simulator.java Aug 18, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.simulators;

import java.rmi.RemoteException;

import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.experiment.Replication;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.jstats.statistics.StatisticsObject;
import nl.tudelft.simulation.language.concurrent.WorkerThread;
import nl.tudelft.simulation.logger.Logger;

/**
 * The Simulator class is an abstract implementation of the SimulatorInterface.
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.13 2004-03-18
 * @since 1.0
 */
public abstract class Simulator extends EventProducer implements
		SimulatorInterface, Runnable
{

	/**
	 * simulatorTime represents the simulationTime
	 * 
	 * @uml.property name="simulatorTime"
	 */
	protected double simulatorTime = Double.NaN;

	/**
	 * running represents the binary state of the simulator
	 * 
	 * @uml.property name="running"
	 */
	protected boolean running = false;

	/**
	 * replication represents the currently active replication
	 * 
	 * @uml.property name="replication"
	 */
	protected Replication replication = null;


	/** a worker */
	protected WorkerThread worker = null;

	/** the simulatorSemaphore */
	protected Object semaphore = new Object();

	/**
	 * constructs a new Simulator
	 */
	public Simulator()
	{
		this.worker = new WorkerThread(this.getClass().getName(), this);
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface
	 *      #getSimulatorTime()
	 * 
	 * @uml.property name="simulatorTime"
	 */
	public double getSimulatorTime()
	{
		return this.simulatorTime;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface
	 *      #getReplication()
	 * 
	 * @uml.property name="replication"
	 */
	public Replication getReplication()
	{
		return this.replication;
	}


	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface
	 *      #initialize(Replication)
	 */
	public void initialize(final Replication replication)
			throws RemoteException, SimRuntimeException
	{
		if (replication == null)
		{
			throw new IllegalArgumentException("replication == null ?");
		}
		if (this.isRunning())
		{
			throw new SimRuntimeException(
					"Cannot initialize a running simulator");
		}
		synchronized (this.semaphore)
		{
			this.removeAllListeners(StatisticsObject.class);
			this.replication = replication;
			this.simulatorTime = 0.0;
			this.fireEvent(SimulatorInterface.START_REPLICATION_EVENT, null,
					((SimulatorInterface) this).getSimulatorTime());
			this.fireEvent(SimulatorInterface.TIME_CHANGED_EVENT,
					this.simulatorTime, this.simulatorTime);
			Logger.finer(this, "initialize", "OK");
		}
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface#isRunning()
	 * 
	 * @uml.property name="running"
	 */
	public boolean isRunning()
	{
		return this.running;
	}

	/**
	 * The run method defines the actual time step mechanism of the simulator.
	 * The implementation of this method depends on the formalism. Where
	 * discrete event formalisms loop over an eventlist continuous simulators
	 * take pre-defined time steps.
	 *  
	 */
	public abstract void run();

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface#start()
	 */
	public void start() throws SimRuntimeException, RemoteException
	{
		if (this.isRunning())
		{
			throw new SimRuntimeException("Cannot start a running simulator");
		}
		if (this.replication == null)
		{
			throw new SimRuntimeException("Cannot start a simulator"
					+ " without replication details");
		}
		if (this.simulatorTime >= this.replication.getRunControl()
				.getRunLength())
		{
			throw new SimRuntimeException("Cannot start simulator : "
					+ "simulatorTime = runLength");
		}
		synchronized (this.semaphore)
		{
			this.running = true;
			this.fireEvent(START_EVENT, null);
			this.fireEvent(SimulatorInterface.TIME_CHANGED_EVENT,
					this.simulatorTime, this.simulatorTime);
			Logger.finer(this, "start", "OK");
			if (!Thread.currentThread().getName().equals(this.worker.getName()))
			{
				this.worker.interrupt();
			} else
			{
				this.run();
			}
		}
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface#step()
	 */
	public void step() throws SimRuntimeException
	{
		if (this.isRunning())
		{
			throw new SimRuntimeException("Cannot step a running simulator");
		}
		if (this.replication == null)
		{
			throw new SimRuntimeException("Cannot step a simulator "
					+ "without replication details");
		}
		if (this.simulatorTime >= this.replication.getRunControl()
				.getRunLength())
		{
			throw new SimRuntimeException("Cannot step simulator: "
					+ "SimulatorTime = runControl.runLength");
		}
		Logger.finer(this, "step", "OK");
		this.fireEvent(SimulatorInterface.STEP_EVENT, null);
	}

	/**
	 * @see nl.tudelft.simulation.dsol.simulators.SimulatorInterface#stop()
	 */
	public void stop()
	{
		if (this.isRunning())
		{
			this.running = false;
			Logger.finer(this, "stop", "OK");
			if (this.simulatorTime >= this.getReplication().getRunControl()
					.getRunLength())
			{
				this
						.fireEvent(new Event(
								SimulatorInterface.END_OF_REPLICATION_EVENT,
								this, null));
			}
			this.fireEvent(SimulatorInterface.STOP_EVENT, null);
		}
	}
}