/*
 * @(#) Counter.java Sep 21, 2003 Copyright (c) 2002-2005 Delft University of
 * Technology Jaffalaan 5, 2628 BX Delft, the Netherlands. All rights reserved.
 * This software is proprietary information of Delft University of Technology
 * The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.statistics;

import java.rmi.RemoteException;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventProducerInterface;
import nl.tudelft.simulation.event.EventType;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.naming.context.ContextUtil;

/**
 * The counter extends the counter and links this it to the dsol framework <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision: 1.7 $ $Date: 2005/08/04 12:09:30 $
 * @author <a href="http://www.peter-jacobs.com">Peter Jacobs </a>
 */
public class Counter extends nl.tudelft.simulation.jstats.statistics.Counter
{

    /** the simulator to subscribe to and from */
    private SimulatorInterface simulator = null;

    /** we stopped the counter */
    private boolean stopped = false;

    /**
     * constructs a new Counter
     * 
     * @param description refers to the description of this counter
     * @param simulator the simulator
     * @throws RemoteException on network failure
     */
    public Counter(final String description, final SimulatorInterface simulator)
            throws RemoteException
    {
        super(description);
        this.simulator = simulator;
        if (this.simulator.getSimulatorTime() > this.simulator.getReplication()
                .getTreatment().getWarmupPeriod())
        {
            this.initialize();
        } else
        {
            this.simulator.addListener(this, SimulatorInterface.WARMUP_EVENT,
                    false);
        }
        this.simulator.addListener(this,
                SimulatorInterface.END_OF_REPLICATION_EVENT, false);
        try
        {
            Context context = ContextUtil.lookup(this.simulator
                    .getReplication().getContext(), "/statistics");
            ContextUtil.bind(context, this);
        } catch (RemoteException exception)
        {
            Logger.warning(this, "<init>", exception);
        }
    }

    /**
     * constructs a new Counter
     * 
     * @param description the description
     * @param simulator the simulator of this model
     * @param target the target on which to count
     * @param field the field which is counted
     * @throws RemoteException on network failure
     */
    public Counter(final String description,
            final SimulatorInterface simulator,
            final EventProducerInterface target, final EventType field)
            throws RemoteException
    {
        this(description, simulator);
        target.addListener(this, field, false);
    }

    /**
     * @see nl.tudelft.simulation.event.EventListenerInterface
     *      #notify(nl.tudelft.simulation.event.EventInterface)
     */
    @Override
    public void notify(final EventInterface event)
    {
        if (this.stopped)
        {
            return;
        }
        try
        {
            if (event.getSource().equals(this.simulator))
            {
                if (event.getType().equals(SimulatorInterface.WARMUP_EVENT))
                {
                    this.simulator.removeListener(this,
                            SimulatorInterface.WARMUP_EVENT);
                    super.initialize();
                    return;
                }
                if (event.getType().equals(
                        SimulatorInterface.END_OF_REPLICATION_EVENT))
                {
                    this.stopped = true;
                    this.simulator.removeListener(this,
                            SimulatorInterface.END_OF_REPLICATION_EVENT);
                    this.endOfReplication();
                    return;
                }
            } else if (this.isInitialized())
            {
                super.notify(event);
            }
        } catch (RemoteException remoteException)
        {
            Logger.warning(this, "notify", remoteException);
        }
    }

    /**
     * endOfReplication is invoked to store the final results
     */
    protected void endOfReplication()
    {
        try
        {
            String[] parts = nl.tudelft.simulation.naming.context.ContextUtil
                    .resolveKey(this).split("/");
            String key = "";
            for (int i = 0; i < parts.length; i++)
            {
                if (i != parts.length - 2)
                {
                    key = key + parts[i] + "/";
                }
            }
            key = key.substring(0, key.length() - 1);
            nl.tudelft.simulation.jstats.statistics.Tally tally = null;
            try
            {
                tally = (nl.tudelft.simulation.jstats.statistics.Tally) new InitialContext()
                        .lookup(key);
            } catch (NamingException exception)
            {
                tally = new nl.tudelft.simulation.jstats.statistics.Tally(
                        this.description);
                new InitialContext().bind(key, tally);
                tally.initialize();
            }
            tally.notify(new Event(null, this, new Long(this.count)));
        } catch (Exception exception)
        {
            Logger.warning(this, "endOfReplication", exception);
        }
    }
}