/*
 * @(#) DSOLApplication.java Oct 24, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.gui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Properties;
import java.util.TreeSet;

import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.windows.DSOLFrame;
import nl.tudelft.simulation.dsol.simulators.Animator;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.language.io.URLResource;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.logger.handlers.MemoryHandler;

/**
 * The DSOL Application <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 1.0 18.10.2003 <br>
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 */
public class DSOLApplication extends EventProducer implements
		DSOLApplicationInterface
{
	/**
	 * the experiment
	 * 
	 * @uml.property name="experiment"
	 */
	protected Experiment experiment = null;

	/**
	 * the properties
	 * 
	 * @uml.property name="properties"
	 */
	protected Properties properties = new Properties();


	/** the recents */
	protected TreeSet recents = new TreeSet(new RecentComparator());

	/**
	 * constructs a new DSOLApplicationInterface
	 */
	public DSOLApplication()
	{
		super();
		Runtime.runFinalizersOnExit(true);
		Logger.setDefaultHandler(MemoryHandler.class);
		try
		{
			this.readProperties(this.properties);
			new DSOLFrame(this);
		} catch (Exception exception)
		{
			Logger.warning(this, "DSOLApplicationInterface", exception);
		}
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable
	{
		this.writeProperties(this.properties);
		super.finalize();
	}

	/**
	 * returns the experiment
	 * 
	 * @return Experiment
	 * 
	 * @uml.property name="experiment"
	 */
	public Experiment getExperiment()
	{
		return this.experiment;
	}

	/**
	 * sets the experiment
	 * 
	 * @param experiment the experiment
	 * 
	 * @uml.property name="experiment"
	 */
	public void setExperiment(final Experiment experiment)
	{
		this.experiment = experiment;
		if (this.experiment != null && this.experiment.getUrl() != null)
		{
			this.recents.add(new Recent(experiment.getUrl().toString(), System
					.currentTimeMillis()));
			Object[] elements = this.recents.toArray();
			this.recents.clear();
			this.recents.addAll(Arrays.asList(elements));
			try
			{
				this.writeProperties(this.properties);

				this.experiment.setSimulator(new Animator());
				this.experiment.start();
				this.experiment.getSimulator().stop();
			} catch (Exception e)
			{
				e.printStackTrace();
			}
		}
		this.fireEvent(EXPERIMENT_CHANGED_EVENT, experiment);
	}

	/**
	 * returns the properties
	 * 
	 * @return Properties
	 * 
	 * @uml.property name="properties"
	 */
	public Properties getProperties()
	{
		return this.properties;
	}

	/**
	 * executes the dsol control panel
	 * 
	 * @param args the command-line arguments
	 */
	public static void main(final String[] args)
	{
		new DSOLApplication();
	}

	/**
	 * reads the properties
	 * 
	 * @param properties the properties to read
	 * @throws IOException on failure
	 */
	private void readProperties(final Properties properties) throws IOException
	{
		String tempPath = System.getProperty("java.io.tmpdir");
		String fileName = this.getClass().getName() + "_dsol.properties";
		File file = new File(tempPath, fileName);
		if (file.exists())
		{
			FileInputStream stream = new FileInputStream(file);
			properties.load(stream);
			stream.close();
		} else
		{
			properties
					.load(URLResource.getResourceAsStream("/dsol.properties"));
		}
		int i = 0;
		String value = null;
		while ((value = properties.getProperty("recent[" + i + "]")) != null)
		{
			String[] elements = value.split("!#!");
			this.recents.add(new Recent(elements[0], Long
					.parseLong(elements[1])));
			i++;
		}
	}

	/**
	 * reads the properties
	 * 
	 * @param properties the properties to read
	 * @throws IOException on failure
	 */
	private void writeProperties(final Properties properties) throws IOException
	{
		int loop = 0;
		for (Iterator i = this.recents.iterator(); i.hasNext();)
		{
			Recent recent = (Recent) i.next();
			if (loop <= 5)
			{
				this.properties.put("recent[" + loop + "]", recent.getLabel()
						+ "!#!" + recent.getExecutionTime());
			}
			loop++;
		}
		String tempPath = System.getProperty("java.io.tmpdir");
		String fileName = this.getClass().getName() + "_dsol.properties";
		File file = new File(tempPath, fileName);
		if (!file.exists())
		{
			file.createNewFile();
		}
		FileOutputStream outputStream = new FileOutputStream(file);
		this.properties.store(outputStream, "");
		outputStream.close();
	}

	/**
	 * the Recent Comparator
	 */
	private static class RecentComparator implements Comparator
	{
		/**
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(final Object o1, final Object o2)
		{
			Recent r1 = (Recent) o1;
			Recent r2 = (Recent) o2;
			if (r1.getLabel().equals(r2.getLabel()))
			{
				long executionTime = Math.max(r1.getExecutionTime(), r2
						.getExecutionTime());
				r1.executionTime = executionTime;
				r2.executionTime = executionTime;
				return 0;
			}
			return new Long(r2.getExecutionTime()).compareTo(new Long(r1
					.getExecutionTime()));
		}
	}

	/**
	 * A RecentItem
	 */
	private static class Recent
	{
		/** the label to use */
		private String label = null;

		/** the executionTime */
		protected long executionTime = 0L;

		/**
		 * constructs a new Recent
		 * 
		 * @param label the label of the experimen
		 * @param executionTime the executionTime of the experiment
		 */
		public Recent(final String label, final long executionTime)
		{
			this.executionTime = executionTime;
			this.label = label;
		}

		/**
		 * @return Returns the executionTime.
		 */
		public long getExecutionTime()
		{
			return this.executionTime;
		}

		/**
		 * @return Returns the label.
		 */
		public String getLabel()
		{
			return this.label;
		}
	}
}