/*
 * @(#) Animation3DCanvas.java May 10, 2004
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.gui.animation3D;

import java.rmi.RemoteException;

import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.Locale;
import javax.media.j3d.VirtualUniverse;
import javax.naming.Binding;
import javax.naming.NamingEnumeration;
import javax.naming.event.EventContext;
import javax.naming.event.NamespaceChangeListener;
import javax.naming.event.NamingEvent;
import javax.naming.event.NamingExceptionEvent;

import nl.tudelft.simulation.dsol.animation.D3.Renderable3DInterface;
import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.animation3D.mouse.PickObjectMouseBehavior;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.naming.InitialEventContext;

import com.sun.j3d.utils.universe.SimpleUniverse;

/**
 * Animation3DCanvas, a canvas for 3d animation <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 1.0 10.05.2004 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/royc/index.htm">Roy Chin
 *         </a>
 */
public class Animation3DCanvas extends Canvas3D implements
		EventListenerInterface, NamespaceChangeListener
{

	/** the application */
	private DSOLApplicationInterface application = null;

	/** the eventContext */
	private EventContext context = null;

	/**
	 * View branch
	 * 
	 * @uml.property name="viewBranch"
	 */
	private ViewBranch viewBranch = null;

	/**
	 * Content branch
	 * 
	 * @uml.property name="contentBranch"
	 */
	private ContentBranch contentBranch = null;


	/**
	 * Constructs a new panel
	 * 
	 * @param application DSOLApplicationInterface
	 */
	public Animation3DCanvas(final DSOLApplicationInterface application)
	{
		super(SimpleUniverse.getPreferredConfiguration());
		this.application = application;
		initializePanel();
		initializeSubscription();
	}

	/**
	 * Initializes the panel
	 */
	public void initializePanel()
	{
		// ----------------
		// Configure canvas
		this.setFocusable(true);

		// ------------------------------------
		// Create a default universe and locale
		VirtualUniverse universe = new VirtualUniverse();
		Locale locale = new Locale(universe);

		// ---------------------
		// Build the scene graph
		this.viewBranch = new ViewBranch(this);
		this.contentBranch = new ContentBranch();

		// --------------
		// Enable picking
		new PickObjectMouseBehavior(this, this.contentBranch,
				new BoundingSphere());

		locale.addBranchGraph(this.viewBranch);
		locale.addBranchGraph(this.contentBranch);
	}

	/**
	 * initializes the panel subscription
	 */
	private void initializeSubscription()
	{
		try
		{
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
			this.notify(new Event(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT,
					this.application, this.application.getExperiment()));
		} catch (RemoteException exception)
		{
			Logger.warning(this, "initialize", exception);
		}
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface#notify
	 *      (nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event) throws RemoteException
	{
		// UPDATE_ANIMATION_EVENT
		if (event.getSource() instanceof AnimatorInterface
				&& event.getType().equals(
						AnimatorInterface.UPDATE_ANIMATION_EVENT))
		{
			if (this.getWidth() > 0 || this.getHeight() > 0)
			{
				this.updateElements();
				//this.repaint();
			}
			return;
		}
		// START_REPLICATION_EVENT
		if (event.getSource() instanceof AnimatorInterface
				&& event.getType().equals(
						SimulatorInterface.START_REPLICATION_EVENT))
		{
			//System.out.println("START_REPLICATION_EVENT");
			this.contentBranch.getDynamicObjectBranch().removeAllChildren();
			this.contentBranch.getStaticObjectBranch().removeAllChildren();
			AnimatorInterface simulator = (AnimatorInterface) event.getSource();
			String contextName = this.application.getExperiment().getRun()
					+ "/treatment("
					+ simulator.getReplication().getRunControl().getTreatment()
							.getNumber() + ")/replication("
					+ simulator.getReplication().getNumber() + ")/animation/3D";
			try
			{
				if (this.context != null)
				{
					this.context.removeNamingListener(this);
				}
				this.context = (EventContext) new InitialEventContext()
						.lookup(contextName);
				this.context.addNamingListener("", EventContext.SUBTREE_SCOPE,
						this);
				NamingEnumeration list = this.context.listBindings("");
				while (list.hasMore())
				{
					Binding binding = (Binding) list.next();
					this.objectAdded(new NamingEvent(this.context, -1, binding,
							binding, null));
				}
				this.updateElements();
				//this.repaint();
			} catch (Exception exception)
			{
				Logger.warning(this, "notify", exception);
			}
		}
		// SIMULATOR_CHANGED_EVENT
		if (event.getSource() instanceof Experiment
				&& event.getType().equals(Experiment.SIMULATOR_CHANGED_EVENT))
		{
			//System.out.println("SIMULATOR_CHANGED_EVENT");
			this.contentBranch.getDynamicObjectBranch().removeAllChildren();
			this.contentBranch.getStaticObjectBranch().removeAllChildren();
			this.updateElements();
			//this.repaint();
			if (event.getContent() instanceof AnimatorInterface
					&& event.getContent() != null)
			{
				AnimatorInterface simulator = (AnimatorInterface) event
						.getContent();
				simulator.addListener(this,
						AnimatorInterface.UPDATE_ANIMATION_EVENT);
				if (simulator.getReplication() != null)
				{
					this.notify(new Event(
							SimulatorInterface.START_REPLICATION_EVENT,
							simulator, simulator));
				} else
				{
					simulator.addListener(this,
							SimulatorInterface.START_REPLICATION_EVENT);
				}
			}
		}
		// EXPERIMENT_CHANGED_EVENT
		if (event.getSource().equals(this.application)
				&& event.getType().equals(
						DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT))
		{
			//System.out.println("EXPERIMENT_CHANGED_EVENT");
			this.contentBranch.getDynamicObjectBranch().removeAllChildren();
			this.contentBranch.getStaticObjectBranch().removeAllChildren();
			this.updateElements();
			//this.repaint();
			if (event.getContent() != null)
			{
				Experiment experiment = (Experiment) event.getContent();
				experiment
						.addListener(this, Experiment.SIMULATOR_CHANGED_EVENT);
				this.notify(new Event(Experiment.SIMULATOR_CHANGED_EVENT,
						experiment, experiment.getSimulator()));
			}
			return;
		}

	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener#objectAdded
	 *      (javax.naming.event.NamingEvent)
	 */
	public void objectAdded(final NamingEvent namingEvent)
	{
		Renderable3DInterface element = (Renderable3DInterface) namingEvent
				.getNewBinding().getObject();
		// Add the element's model to the content branch
		BranchGroup model = (BranchGroup) element;
		if (model != null)
		{
			if (element.getType() != Renderable3DInterface.DYNAMIC_OBJECT)
			{
				this.contentBranch.getStaticObjectBranch().addChild(model);
			} else
			{
				this.contentBranch.getDynamicObjectBranch().addChild(model);
			}
		}
	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener#objectRemoved
	 *      (javax.naming.event.NamingEvent)
	 */
	public void objectRemoved(final NamingEvent namingEvent)
	{
		System.out.println("Animation3d: Object removed");
		Renderable3DInterface element = (Renderable3DInterface) namingEvent
				.getOldBinding().getObject();

		// Remove element's model from the branch graph
		this.contentBranch.getDynamicObjectBranch().removeChild(
				(BranchGroup) element);
	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener#objectRenamed
	 *      (javax.naming.event.NamingEvent)
	 */
	public void objectRenamed(final NamingEvent namingEvent)
	{
		this.objectRemoved(namingEvent);
		this.objectAdded(namingEvent);
	}

	/**
	 * @see javax.naming.event.NamingListener#namingExceptionThrown
	 *      (javax.naming.event.NamingExceptionEvent)
	 */
	public void namingExceptionThrown(final NamingExceptionEvent namingEvent)
	{
		Logger.warning(this, "namingExceptionThrown", namingEvent
				.getException());
	}

	/**
	 * Update all elements
	 */
	public void updateElements()
	{
		BranchGroup dynamicObjectBranch = this.contentBranch
				.getDynamicObjectBranch();
		// Synchronize is meant to prevent shaky animation
		// when updating lots of objects. Seems to work.
		synchronized (this)
		{
			for (int i = 0; i < dynamicObjectBranch.numChildren(); i++)
			{
				Renderable3DInterface element = (Renderable3DInterface) dynamicObjectBranch
						.getChild(i);
				element.update();
			}
		}
	}

	/**
	 * @return ContentBranch
	 * 
	 * @uml.property name="contentBranch"
	 */
	public ContentBranch getContentBranch()
	{
		return this.contentBranch;
	}

	/**
	 * @return ViewBranch
	 * 
	 * @uml.property name="viewBranch"
	 */
	public ViewBranch getViewBranch()
	{
		return this.viewBranch;
	}

}