/*
 * @(#) EditorInputListener.java Mar 2, 2004
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.gui.editor2D.mouse;

import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;

import nl.tudelft.simulation.dsol.animation.Editable;
import nl.tudelft.simulation.dsol.animation.D2.EditableRenderable2DInterface;
import nl.tudelft.simulation.dsol.animation.D2.Renderable2DInterface;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.animation2D.AnimationPanel;
import nl.tudelft.simulation.dsol.gui.animation2D.mouse.InputListener;
import nl.tudelft.simulation.dsol.gui.editor2D.Editor2DPanel;
import nl.tudelft.simulation.dsol.gui.editor2D.actions.EditorUtilities;
import nl.tudelft.simulation.introspection.gui.IntroSpectionDialog;
import nl.tudelft.simulation.language.d3.CartesianPoint;

/**
 * An input listener for the editor
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 1.0 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/royc/index.htm">Roy Chin
 *         </a>
 */
public class EditorInputListener extends InputListener
{
    /** the source panel */
    protected Editor2DPanel panel = null;

    /** did we click within the selected object? */
    protected boolean clickedWithinSelected = false;

    /** Last mouse button that was pressed */
    protected int mouseButton = 0;

    /** Point where the mouse is when dragging */
    protected Point2D dragCoord = null;

    /**
     * constructor
     * 
     * @param application
     *            the application
     * @param panel
     *            the panel
     */
    public EditorInputListener(final DSOLApplicationInterface application,
            final AnimationPanel panel)
    {
        super(application, panel);
        this.panel = (Editor2DPanel) panel;
    }

    /**
     * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
     */
    public void mouseClicked(final MouseEvent event)
    {
        this.panel.requestFocus();
    }

    /**
     * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
     */
    public void mouseDragged(final MouseEvent event)
    {
        super.mouseDragged(event);

        Point2D newCoordinate = Renderable2DInterface.Util.getWorldCoordinates(
                event.getPoint(), this.panel.getExtent(), this.panel.getSize());
        Point2D centerCoordinate = Renderable2DInterface.Util
                .getWorldCoordinates(mouseClicked, this.panel.getExtent(),
                        this.panel.getSize());
        Point2D oldCoordinate = Renderable2DInterface.Util.getWorldCoordinates(
                this.dragCoord, this.panel.getExtent(), this.panel.getSize());

        if ((this.mouseButton == MouseEvent.BUTTON1)
                && (this.panel.getSelectedMode() == Editor2DPanel.MODE_EDIT)
                && (this.panel.getSelectedPoint() == null)
                && (this.clickedWithinSelected))
        {
            if (this.panel.getSelectedEditMode() == Editor2DPanel.EDIT_MODE_MOVE)
            {
                // Move the entire shape
                EditorUtilities.moveEditable(this.panel
                        .getSelectedEditableRenderable(), newCoordinate,
                        oldCoordinate);
            } else if (this.panel.getSelectedEditMode() == Editor2DPanel.EDIT_MODE_ROTATE)
            {
                // Rotate the entire shape
                EditorUtilities.rotateEditable(this.panel
                        .getSelectedEditableRenderable(), newCoordinate,
                        centerCoordinate, oldCoordinate);
            }
        }

        this.dragCoord = (Point2D) event.getPoint().clone();
    }

    /**
     * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
     */
    public void mousePressed(final MouseEvent event)
    {
        super.mousePressed(event);

        this.mouseButton = event.getButton();
        this.dragCoord = event.getPoint();

        // What to do in which mode
        if (this.panel.getSelectedMode() == Editor2DPanel.MODE_EDIT)
        {
            CartesianPoint selectedPoint = this
                    .determineSelectedVertice(this.mouseClicked);
            this.panel.setSelectedPoint(selectedPoint);
            if (this.panel.getSelectedPoint() == null)
            {
                this.clickedWithinSelected = this.isInside(this.mouseClicked);
            }
        }
    }

    /**
     * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
     */
    public void mouseReleased(final MouseEvent event)
    {
        super.mouseReleased(event);

        switch (this.mouseButton) 
        {
        case MouseEvent.BUTTON1:
            this.performLeftMouseReleasedAction(event);
            break;
        case MouseEvent.BUTTON3:
            this.performRightMouseReleasedAction(event);
            break;
        default:
        }
        this.panel.repaint();
    }

    /**
     * perform one of the actions of the editor
     * 
     * @param event
     *            mouse event
     */
    protected void performLeftMouseReleasedAction(final MouseEvent event)
    {
        Point2D point = event.getPoint();
        Point2D newCoordinate = Renderable2DInterface.Util.getWorldCoordinates(
                point, this.panel.getExtent(), this.panel.getSize());
        Point2D oldCoordinate = Renderable2DInterface.Util.getWorldCoordinates(
                this.dragCoord, this.panel.getExtent(), this.panel.getSize());
        Point2D centerCoordinate = Renderable2DInterface.Util
                .getWorldCoordinates(mouseClicked, this.panel.getExtent(),
                        this.panel.getSize());
        switch (this.panel.getSelectedMode()) 
        {
        case Editor2DPanel.MODE_SELECT:
            EditorUtilities.selectEditable(newCoordinate, this.panel);
            break;
        case Editor2DPanel.MODE_NEW:
            EditorUtilities.instantiateNewEditable(newCoordinate,
                    this.application, this.panel);
            break;
        case Editor2DPanel.MODE_EDIT:
            EditableRenderable2DInterface selected = this.panel
                    .getSelectedEditableRenderable();
            if (selected != null)
            {
                if (this.panel.getSelectedPoint() != null)
                {
                    // Move selected point
                    EditorUtilities.moveSelectedPoint(selected, newCoordinate,
                            this.panel);
                } else if (this.clickedWithinSelected)
                {
                    if (this.panel.getSelectedEditMode() == Editor2DPanel.EDIT_MODE_MOVE)
                    {
                        // Move the entire shape
                        EditorUtilities.moveEditable(selected, newCoordinate,
                                oldCoordinate);
                    }
                    if (this.panel.getSelectedEditMode() == Editor2DPanel.EDIT_MODE_ROTATE)
                    {
                        // Rotate the entire shape
                        EditorUtilities.rotateEditable(selected, newCoordinate,
                                centerCoordinate, oldCoordinate);
                    }
                }
            }
            break;
        default:
        }
    }

    /**
     * what to do if the right mouse button was released
     * 
     * @param event
     *            MouseEvent
     */
    protected void performRightMouseReleasedAction(final MouseEvent event)
    {
        Point2D world = Renderable2DInterface.Util.getWorldCoordinates(event
                .getPoint(), this.panel.getExtent(), this.panel.getSize());
        if (this.panel.getSelectedMode() == Editor2DPanel.MODE_SELECT)
        {
            // Determine which object was selected, if any, and
            // show the introspection window for it.
            Renderable2DInterface selected = EditorUtilities.selectEditable(
                    world, this.panel);
            if (selected != null)
            {
                new IntroSpectionDialog(selected.getSource(), selected
                        .getSource().toString());
            }
        } else if (this.panel.getSelectedMode() == Editor2DPanel.MODE_EDIT)
        {
            if (event.isPopupTrigger())
            {
                this.popup(event);
                return;
            }
        }
    }

    /**
     * determine if a vertice was selected
     * 
     * @param mousePosition
     *            the position of the mouse cursor
     * @return the selected vertice
     */
    private CartesianPoint determineSelectedVertice(final Point2D mousePosition)
    {
        CartesianPoint selectedPoint = null;
        this.panel.setSelectedPoint(null);
        EditableRenderable2DInterface selected = this.panel
                .getSelectedEditableRenderable();
        if (selected == null)
        {
            return null;
        }
        Editable editable = (Editable) selected.getSource();
        CartesianPoint[] vertices = editable.getVertices();
        CartesianPoint[] positions = new CartesianPoint[vertices.length];
        for (int i = 0; i < vertices.length; i++)
        {
            positions[i] = EditorUtilities.convertToGlobalCoordinates(
                    vertices[i], editable.getLocation());
            Point2D p1 = Renderable2DInterface.Util.getScreenCoordinates(
                    new Point2D.Double(positions[i].x, positions[i].y),
                    this.panel.getExtent(), this.panel.getSize());
            if (Math.abs(mousePosition.getX() - (int) p1.getX()) <= Editor2DPanel.PLACEHOLDERSIZE)
            {
                if (Math.abs(mousePosition.getY() - (int) p1.getY()) <= Editor2DPanel.PLACEHOLDERSIZE)
                {
                    selectedPoint = vertices[i];
                }
            }
        }
        return selectedPoint;
    }

    /**
     * determine if the mouse pointer is inside a renderable
     * 
     * @param mousePosition
     *            the position of the mouse cursor
     * @return true if inside
     */
    private boolean isInside(final Point2D mousePosition)
    {
        EditableRenderable2DInterface selected = this.panel
                .getSelectedEditableRenderable();
        if (selected == null)
        {
            return false;
        }
        Point2D worldCoord = Renderable2DInterface.Util
                .getWorldCoordinates(this.mouseClicked, this.panel.getExtent(),
                        this.panel.getSize());
        if (((Renderable2DInterface) selected).contains(worldCoord, this.panel
                .getExtent(), this.panel.getSize()))
        {
            return true;
        }
        return false;

    }
}