/*
 * @(#)TablePanel.java Sep 21, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.gui.panels;

import info.clearthought.layout.TableLayout;
import info.clearthought.layout.TableLayoutConstants;

import java.awt.Component;
import java.awt.Point;

import javax.swing.JPanel;
import javax.swing.table.DefaultTableModel;


/**
 * The tablePanel provides an easy to access table for the DSOL Framework <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>
 */
public class TablePanel extends JPanel
{
	/** the content of the tablePanel */
	private DefaultTableModel content = null;

	/**
	 * Constructor for TablePanel.
	 * 
	 * @param columns the amount of columns
	 * @param rows the amount of rows
	 */
	public TablePanel(final int rows, final int columns)
	{
		super();
		this.content = new DefaultTableModel(rows, columns);
	}

	/**
	 * adds a row to the tablePanel
	 */
	public void addRow()
	{
		this.content.addRow((Object[]) null);
		this.validate();
	}

	/**
	 * adds a column to the tablePanel
	 */
	public void addColumn()
	{
		this.content.addColumn((Object[]) null);
		this.validate();
	}

	/**
	 * deletes a row from the contentpanel
	 */
	public void deleteRow()
	{
		if (this.content.getRowCount() == 1)
		{
			return;
		}
		this.content.removeRow(this.content.getRowCount() - 1);
		this.validate();
	}

	/**
	 * deletes a column from the tablePanel
	 */
	public void deleteColumn()
	{
		if (this.content.getColumnCount() == 1)
		{
			return;
		}
		DefaultTableModel result = new DefaultTableModel(this.content
				.getRowCount(), this.content.getColumnCount() - 1);
		for (int rows = 0; rows < this.content.getRowCount(); rows++)
		{
			for (int cols = 0; cols < this.content.getColumnCount() - 1; cols++)
			{
				result.setValueAt(this.content.getValueAt(rows, cols), rows,
						cols);
			}
		}
		this.content = result;
		this.validate();
	}

	/**
	 * returns the row of a specific point
	 * 
	 * @param point the xy-point
	 * @return int the row
	 */
	public int getRow(final Point point)
	{
		double heigth = this.getHeight() / (double) this.getRows();
		return (int) Math.floor(point.getY() / heigth);
	}

	/**
	 * returns the column of a point on the panel
	 * 
	 * @param point the x-y point
	 * @return int the column
	 */
	public int getColumn(final Point point)
	{
		double width = this.getWidth() / (double) this.getColumns();
		return (int) Math.floor(point.getX() / width);
	}

	/**
	 * Method setCell.
	 * 
	 * @param container the subject of the cell
	 * @param row of the cell
	 * @param column of the cell
	 */
	public void setCell(final Component container, final int row,
			final int column)
	{
		this.content.setValueAt(container, row, column);
		this.validate();
	}

	/**
	 * contains this container
	 * 
	 * @param container contains the container..
	 * @return boolean
	 */
	public boolean contains(final Component container)
	{
		for (int c = 0; c < this.getColumns(); c++)
		{
			for (int r = 0; r < this.getRows(); r++)
			{
				if (this.getCell(r, c).equals(container))
				{
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * gets the container in a cell
	 * 
	 * @param row the row
	 * @param column the column
	 * @return Component
	 */
	public Component getCell(final int row, final int column)
	{
		double height = this.getHeight() / (double) this.getRows();
		double width = this.getWidth() / (double) this.getColumns();
		Point point = new Point((int) Math.round((column + 0.5) * width),
				(int) Math.round((row + 0.5) * height));
		return this.getComponentAt(point.x, point.y);
	}

	/**
	 * removes the content of a cell
	 * 
	 * @param row the row
	 * @param column the column
	 */
	public void removeCell(final int row, final int column)
	{
		this.remove(this.getCell(row, column));
	}

	/**
	 * Method refractor computes the size
	 * 
	 * @param number refers to the number of rows/columns
	 * @return double[] the result
	 */
	private double[] refractor(final int number)
	{
		double[] result = new double[number];
		for (int i = 0; i < result.length; i++)
		{
			result[i] = 1.0 / number * 1.0;
			if (result[i] == 1.0)
			{
				result[i] = TableLayoutConstants.FILL;
			}
		}
		return result;
	}

	/**
	 * returns the number of Columns
	 * 
	 * @return int
	 */
	public int getColumns()
	{
		return this.content.getColumnCount();
	}

	/**
	 * returns the number of rows
	 * 
	 * @return int the number of rows
	 */
	public int getRows()
	{
		return this.content.getRowCount();
	}

	/**
	 * @see java.awt.Component#validate()
	 */
	public void validate()
	{
		this.removeAll();
		double[][] tableDefinition = {
				this.refractor(this.content.getColumnCount()),
				this.refractor(this.content.getRowCount())};
		TableLayout layout = new TableLayout(tableDefinition);
		this.setLayout(layout);
		for (int rows = 0; rows < this.content.getRowCount(); rows++)
		{
			for (int cols = 0; cols < this.content.getColumnCount(); cols++)
			{
				Component component = (Component) this.content.getValueAt(rows,
						cols);
				if (component != null)
				{
					this.add(component, "" + cols + "," + rows + ",C,C");
				}
			}
		}
		super.validate();
		super.repaint();
	}
}