/*
 * @(#) DSOLFrame.java Oct 13, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.gui.windows;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.io.IOException;
import java.net.URL;

import javax.swing.BorderFactory;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.UIInitializer;
import nl.tudelft.simulation.dsol.gui.menu.FileMenu;
import nl.tudelft.simulation.dsol.gui.menu.HelpMenu;
import nl.tudelft.simulation.dsol.gui.menu.ToolsMenu;
import nl.tudelft.simulation.dsol.gui.menu.WindowMenu;
import nl.tudelft.simulation.dsol.gui.panels.ControlPanel;
import nl.tudelft.simulation.dsol.gui.panels.ExperimentTree;
import nl.tudelft.simulation.dsol.gui.panels.Statusbar;
import nl.tudelft.simulation.dsol.gui.panels.TabbedPane;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.language.io.URLResource;
import nl.tudelft.simulation.logger.Logger;

/**
 * DSOL Frame <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 1.0 18.10.2003 <br>
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 */
public class DSOLFrame extends JFrame implements EventListenerInterface
{
	/** the preferredSize is the preferredSize of the panel */
	public static final Dimension PREFERRED_SIZE = new Dimension(400, 375);

	/** the application */
	protected DSOLApplicationInterface application = null;

	/** tabbedPanel */
	protected TabbedPane tabbedPanel = null;

	/** the experimentTree */
	protected JTree experimentTree = null;

	/** the statusbar */
	protected Statusbar statusbar = null;

	/** the controlPanel */
	protected ControlPanel controlPanel = null;

	/**
	 * constructs a new DSOLFrame
	 * 
	 * @param application the application
	 */
	public DSOLFrame(final DSOLApplicationInterface application)
	{
		super("DSOL main window");

		try
		{
			UIManager
					.setLookAndFeel("com.jgoodies.plaf.plastic.Plastic3DLookAndFeel");

			//The custom DSOL Color Look&Feel
			UIInitializer.initialize();
			Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
			this.setLocation((int) Math.round(screenSize.width * 0.5 - 0.5
					* PREFERRED_SIZE.width), (int) Math.round(screenSize.height
					* 0.33 - 0.5 * PREFERRED_SIZE.height));
			this.application = application;
			this.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
		} catch (Exception exception)
		{
			Logger.warning(this, "DSOLFrame", exception);
		}
		this.initialize();
		this.pack();
		this.setVisible(true);
	}

	/**
	 * initializes the DSOL Frame
	 */
	private void initialize()
	{
		JPanel contentPane = new JPanel();
		contentPane.setPreferredSize(PREFERRED_SIZE);
		contentPane.setLayout(new BorderLayout());
		this.setContentPane(contentPane);

		JMenuBar menuBar = new JMenuBar();
		menuBar.setBorder(BorderFactory.createLineBorder(Color.GRAY));

		menuBar.add(new FileMenu(this.application));
		menuBar.add(new ToolsMenu(this.application));
		menuBar.add(new WindowMenu(this.application));
		menuBar.add(new HelpMenu(this.application));
		this.setJMenuBar(menuBar);

		this.controlPanel = new ControlPanel(this.application);

		this.tabbedPanel = new TabbedPane(SwingConstants.TOP);
		this.tabbedPanel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createEtchedBorder(), BorderFactory
						.createEmptyBorder()));
		contentPane.add(this.tabbedPanel, BorderLayout.CENTER);
		this.statusbar = new Statusbar(this.application);

		this.experimentTree = new JTree();
		this.experimentTree.setEditable(false);
		this.experimentTree.setFocusable(false);

		URL about = URLResource
				.getResource("/nl/tudelft/simulation/dsol/gui/about.html");
		JEditorPane aboutPane = null;
		try
		{
			aboutPane = new JEditorPane(about);
		} catch (IOException exception)
		{
			Logger.warning(this, "initialize", exception);
		}
		aboutPane.setAlignmentY(Component.BOTTOM_ALIGNMENT);
		aboutPane.setEditable(false);
		this.tabbedPanel.add("About", aboutPane);
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event)
	{
		if (event.getSource().equals(this.application)
				&& event.getType().equals(
						DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT))
		{
			this.tabbedPanel.remove("Description");
			this.tabbedPanel.remove("Experiment");
			this.tabbedPanel.remove("Control");
			if (event.getContent() != null)
			{
				Experiment experiment = (Experiment) event.getContent();
				try
				{
					String descriptionName = "/";
					if (experiment.getModel().getClass().getPackage() != null)
					{
						descriptionName = descriptionName
								+ experiment.getModel().getClass().getPackage()
										.getName().replace('.', '/');
					}
					descriptionName = descriptionName + "/package.html";
					URL description = experiment.getModel().getClass()
							.getResource(descriptionName);
					JEditorPane descriptionPane = null;
					descriptionPane = new JEditorPane(description);
					descriptionPane.setAlignmentY(Component.BOTTOM_ALIGNMENT);
					descriptionPane.setEditable(false);
					this.tabbedPanel.add("Description", 0, new JScrollPane(
							descriptionPane));
				} catch (IOException exception)
				{
					Logger
							.fine(this, "initialize",
									"experiment has no package.html describing the experiment");
				}
				this.setTitle("DSOL main window ("
						+ experiment.getProperty(Experiment.EXPERIMENT_NAME)
						+ ")");
				this.experimentTree.setModel(new ExperimentTree(experiment));
				this.tabbedPanel.add("Experiment", 1, new JScrollPane(
						this.experimentTree));
				this.tabbedPanel.add("Control", 2, this.controlPanel);
				this.getContentPane().add(this.statusbar, BorderLayout.SOUTH);
			} else
			{
				this.setTitle("DSOL main window");
				this.getContentPane().remove(this.statusbar);
			}
			this.validate();
			this.repaint();
		}
	}
}