/*
 * @(#) AnimationPanel.java Nov 3, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.gui.animation2D;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.MouseListener;
import java.awt.geom.Rectangle2D;
import java.lang.reflect.Constructor;
import java.rmi.RemoteException;
import java.util.Collections;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.naming.Binding;
import javax.naming.NamingEnumeration;
import javax.naming.event.EventContext;
import javax.naming.event.NamespaceChangeListener;
import javax.naming.event.NamingEvent;
import javax.naming.event.NamingExceptionEvent;
import javax.vecmath.Point4i;

import nl.tudelft.simulation.dsol.animation.D2.Renderable2DInterface;
import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.animation2D.mouse.InputListener;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.naming.InitialEventContext;

/**
 * The AnimationPanel <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 1.0 18.10.2003 <br>
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 */
public class AnimationPanel extends GridPanel implements
		EventListenerInterface, NamespaceChangeListener
{

	/**
	 * the elements of this panel
	 * 
	 * @uml.property name="elements"
	 */
	private SortedSet elements = Collections.synchronizedSortedSet(new TreeSet(
			new Renderable2DComparator()));


	/** the application */
	private DSOLApplicationInterface application = null;

	/** the eventContext */
	private EventContext context = null;

	/**
	 * the grid must be drawn after all other elements. Therefore we must
	 * override the gridPanel.paintGrid.
	 * 
	 * @uml.property name="showGrid"
	 */
	private boolean showGrid = true;

    /** a line that helps the user to see where he is dragging */
    private Point4i dragLine = new Point4i();

    /** enable drag line */
    private boolean dragLineEnabled = false;

	/**
	 * constructs a new AnimationPanel
	 * 
	 * @param extent the extent of the panel
	 * @param size the size of the panel.
	 * @param application the application
	 */
	public AnimationPanel(final Rectangle2D extent, final Dimension size,
			final DSOLApplicationInterface application)
	{
		super(extent, size);
		super.showGrid = false;
		this.application = application;
		String mouseEditorClassName = this.application.getProperties()
				.getProperty("dsol-gui.animation.panel.mouseEditor");
		InputListener listener = new InputListener(
				application, this);
		if (mouseEditorClassName != null)
		{
			try
			{
				Class editorClass = Class.forName(mouseEditorClassName);
				Constructor constructor = editorClass
						.getConstructor(new Class[]{
								DSOLApplicationInterface.class,
								AnimationPanel.class});
				this.addMouseListener((MouseListener) constructor
						.newInstance(new Object[]{application, this}));

			} catch (Exception exception)
			{
				Logger.warning(this, "<init>", "could not load "
						+ mouseEditorClassName);
				listener = new InputListener(application, this);
			}
		} else
		{
			this.addMouseListener(listener);
		}
		this.addMouseMotionListener(listener);
		this.addMouseWheelListener(listener);
		this.addKeyListener(listener);
		this.initialize();
	}

	/**
	 * initializes the panel
	 */
	private void initialize()
	{
		try
		{
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
			this.notify(new Event(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT,
					this.application, this.application.getExperiment()));
		} catch (RemoteException exception)
		{
			Logger.warning(this, "initialize", exception);
		}
	}

	/**
	 * @see javax.swing.JComponent #paintComponent(java.awt.Graphics)
	 */
	public void paintComponent(final Graphics g)
	{
		super.paintComponent(g);
		Graphics2D g2 = (Graphics2D) g;
		synchronized (this.elements)
		{
			for (Iterator i = this.elements.iterator(); i.hasNext();)
			{
				((Renderable2DInterface) i.next()).paint(g2, this.getExtent(),
						this.getSize(), this);
			}
		}
		if (this.showGrid)
		{
			this.drawGrid(g);
		}
		
		// Draw dragline
        if (this.dragLineEnabled)
        {
            g.setColor(Color.BLACK);
            g.drawLine(this.dragLine.w, this.dragLine.x, this.dragLine.y,
                    this.dragLine.z);
            this.dragLineEnabled = false;
        }
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event) throws RemoteException
	{
		if (event.getSource() instanceof AnimatorInterface
				&& event.getType().equals(
						AnimatorInterface.UPDATE_ANIMATION_EVENT))
		{
			if (this.getWidth() > 0 || this.getHeight() > 0)
			{
				this.repaint();
			}
			return;
		}
		if (event.getSource() instanceof AnimatorInterface
				&& event.getType().equals(
						SimulatorInterface.START_REPLICATION_EVENT))
		{
			this.elements.clear();
			AnimatorInterface simulator = (AnimatorInterface) event.getSource();
			String contextName = this.application.getExperiment().getRun()
					+ "/treatment("
					+ simulator.getReplication().getRunControl().getTreatment()
							.getNumber() + ")/replication("
					+ simulator.getReplication().getNumber() + ")/animation/2D";
			try
			{
				if (this.context != null)
				{
					this.context.removeNamingListener(this);
				}
				this.context = (EventContext) new InitialEventContext()
						.lookup(contextName);
				this.context.addNamingListener("", EventContext.SUBTREE_SCOPE,
						this);
				NamingEnumeration list = this.context.listBindings("");
				while (list.hasMore())
				{
					Binding binding = (Binding) list.next();
					this.objectAdded(new NamingEvent(this.context, -1, binding,
							binding, null));
				}
				this.repaint();
			} catch (Exception exception)
			{
				Logger.warning(this, "notify", exception);
			}
		}
		if (event.getSource() instanceof Experiment
				&& event.getType().equals(Experiment.SIMULATOR_CHANGED_EVENT))
		{
			this.elements.clear();
			this.repaint();
			if (event.getContent() instanceof AnimatorInterface
					&& event.getContent() != null)
			{
				AnimatorInterface simulator = (AnimatorInterface) event
						.getContent();
				simulator.addListener(this,
						AnimatorInterface.UPDATE_ANIMATION_EVENT);
				if (simulator.getReplication() != null)
				{
					this.notify(new Event(
							SimulatorInterface.START_REPLICATION_EVENT,
							simulator, simulator));
				} else
				{
					simulator.addListener(this,
							SimulatorInterface.START_REPLICATION_EVENT);
				}
			}
		}
		if (event.getSource().equals(this.application)
				&& event.getType().equals(
						DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT))
		{
			this.elements.clear();
			this.repaint();
			if (event.getContent() != null)
			{
				Experiment experiment = (Experiment) event.getContent();
				experiment
						.addListener(this, Experiment.SIMULATOR_CHANGED_EVENT);
				this.notify(new Event(Experiment.SIMULATOR_CHANGED_EVENT,
						experiment, experiment.getSimulator()));
			}
			return;
		}
	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener
	 *      #objectAdded(javax.naming.event.NamingEvent)
	 */
	public void objectAdded(final NamingEvent namingEvent)
	{
		Renderable2DInterface element = (Renderable2DInterface) namingEvent
				.getNewBinding().getObject();
		this.elements.add(element);
	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener
	 *      #objectRemoved(javax.naming.event.NamingEvent)
	 */
	public void objectRemoved(final NamingEvent namingEvent)
	{
		Renderable2DInterface element = (Renderable2DInterface) namingEvent
				.getOldBinding().getObject();
		this.elements.remove(element);
	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener
	 *      #objectRenamed(javax.naming.event.NamingEvent)
	 */
	public void objectRenamed(final NamingEvent namingEvent)
	{
		this.objectRemoved(namingEvent);
		this.objectAdded(namingEvent);
	}

	/**
	 * @see javax.naming.event.NamingListener
	 *      #namingExceptionThrown(javax.naming.event.NamingExceptionEvent)
	 */
	public void namingExceptionThrown(final NamingExceptionEvent namingEvent)
	{
		Logger.warning(this, "namingExceptionThrown", namingEvent
				.getException());
	}

	/**
	 * @return Returns the elements.
	 * 
	 * @uml.property name="elements"
	 */
	public SortedSet getElements()
	{
		return this.elements;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.gui.animation2D.GridPanel#isShowGrid()
	 * 
	 * @uml.property name="showGrid"
	 */
	public boolean isShowGrid()
	{
		return this.showGrid;
	}

	/**
	 * @see nl.tudelft.simulation.dsol.gui.animation2D.GridPanel#showGrid(boolean)
	 */
	public synchronized void showGrid(final boolean bool)
	{
		this.showGrid = bool;
		this.repaint();
	}
	
    /**
     * @return returns the dragLine.
     */
    public Point4i getDragLine()
    {
        return this.dragLine;
    }

    /**
     * @return returns the dragLineEnabled.
     */
    public boolean isDragLineEnabled()
    {
        return this.dragLineEnabled;
    }

    /**
     * @param dragLineEnabled
     *            the dragLineEnabled to set.
     */
    public void setDragLineEnabled(final boolean dragLineEnabled)
    {
        this.dragLineEnabled = dragLineEnabled;
    }
}