/*
 * @(#) ShowGridAction.java Oct 24, 2003
 * 
 * Copyright (c) 2002-2005 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands. All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.actions;

import java.rmi.RemoteException;

import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.simulators.DESSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.logger.Logger;

/**
 * The TimestepSliderAction <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision$ $Date$
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja">Peter Jacobs </a>
 */
public class TimeStepSlider extends JSlider implements EventListenerInterface,
		ChangeListener
{
	/** the animator */
	private DESSSimulatorInterface simulator;

	/** fired */
	private boolean fired = false;

	/**
	 * constructs a new ShowGridAction
	 * 
	 * @param application the application
	 */
	public TimeStepSlider(final DSOLApplicationInterface application)
	{
		super(0, 700, 100);
		this.setEnabled(false);
		this.addChangeListener(this);
		try
		{
			application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
			this.notify(new Event(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT,
					application, application.getExperiment()));
		} catch (RemoteException exception)
		{
			Logger.warning(this, "TimeStepSlider", exception);
		}
	}

	/**
	 * @see javax.swing.event.ChangeListener
	 *      #stateChanged(javax.swing.event.ChangeEvent)
	 */
	public void stateChanged(final ChangeEvent event)
	{
		if (this.simulator == null)
		{
			return;
		}
		if (this.fired)
		{
			this.fired = false;
			return;
		}
		this.fired = true;
		try
		{
			boolean shoudPause = this.simulator.isRunning();
			JSlider slider = (JSlider) event.getSource();
			double value = this.getTimeStep(slider.getValue());
			if (shoudPause)
			{
				this.simulator.stop();
			}
			this.simulator.setTimeStep(0.01 * value);
			if (shoudPause)
			{
				this.simulator.start();
			}
			this.validate();
			this.repaint();
		} catch (Exception exception)
		{
			Logger.warning(this, "stateChanged", exception);
		}
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event) throws RemoteException
	{
		if (event.getType().equals(
				DESSSimulatorInterface.TIME_STEP_CHANGED_EVENT)
				&& event.getSource() instanceof SimulatorInterface)
		{
			this.setValue(this.getSliderValue(((Double) event.getContent())
					.doubleValue()));
		}
		if (event.getType().equals(
				DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT)
				&& event.getSource() instanceof DSOLApplicationInterface)
		{
			if (event.getContent() != null)
			{
				Experiment experiment = (Experiment) event.getContent();
				experiment
						.addListener(this, Experiment.SIMULATOR_CHANGED_EVENT);
				this.notify(new Event(Experiment.SIMULATOR_CHANGED_EVENT,
						experiment, experiment.getSimulator()));
			}
			return;
		}
		if (event.getType().equals(Experiment.SIMULATOR_CHANGED_EVENT)
				&& event.getSource() instanceof Experiment)
		{
			if (event.getContent() != null)
			{
				if (event.getContent() instanceof DESSSimulatorInterface)
				{
					this.simulator = (DESSSimulatorInterface) event
							.getContent();
					this.notify(new Event(
							DESSSimulatorInterface.TIME_STEP_CHANGED_EVENT,
							this.simulator, new Double(this.simulator
									.getTimeStep())));
					this.simulator.addListener(this,
							DESSSimulatorInterface.TIME_STEP_CHANGED_EVENT);
					this.setEnabled((true));
				}
			}
		}
	}

	/**
	 * returns the sliderValue based on the timeStep
	 * 
	 * @param timeStep the simulator timeStep
	 * @return the sliderValue
	 */
	public int getSliderValue(final double timeStep)
	{
		int sliderValue = (int) (600 + Math.round(100 * Math.log(timeStep)
				/ Math.log(10)));
		return sliderValue;
	}

	/**
	 * gets the TimeStep value based on the slider value
	 * 
	 * @param value the slider value
	 * @return the corresponding timeStep value
	 */
	public double getTimeStep(final int value)
	{
		double timeStep = Math.pow(10, (value - 600) / 100.0);
		return timeStep;
	}
}