/*
 * $RCSfile: ZoomMouseBehavior.java,v $ A modification of MouseZoom.java *
 * 
 * Copyright (c) 2004 Sun Microsystems, Inc. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met: -
 * Redistribution of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer. - Redistribution in binary
 * form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS
 * LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A
 * RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 * IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT
 * OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY,
 * ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS
 * BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed, licensed or intended for
 * use in the design, construction, operation or maintenance of any nuclear
 * facility.
 * 
 * $Revision: 1.5 $ $Date: 2004/12/16 13:18:56 $ $State: Exp $
 */
/*
 * Modified to zoom along the rotated z-axis (actually move). Copyright (c)
 * 2002-2005 Delft University of Technology Jaffalaan 5, 2628 BX Delft, the
 * Netherlands All rights reserved.
 */
package nl.tudelft.simulation.dsol.gui.animation3D.mouse;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.event.MouseEvent;
import java.util.Enumeration;

import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnAWTEvent;
import javax.media.j3d.WakeupOnBehaviorPost;
import javax.vecmath.Vector3d;

import com.sun.j3d.utils.behaviors.mouse.MouseBehavior;
import com.sun.j3d.utils.behaviors.mouse.MouseBehaviorCallback;

/**
 * ZoomMouseBehavior is a Java3D behavior object that lets users control the Z
 * axis translation of an object via a mouse drag motion with the second mouse
 * button. See MouseRotate for similar usage info. Based on
 * com.sun.j3d.utils.behaviors.mouse.MouseZoom.
 */
public class ZoomMouseBehavior extends MouseBehavior
{
	/** zoom factor */
	private double zFactor = .04;

	/** translation */
	private Vector3d translation = new Vector3d();

	/** callback */
	private MouseBehaviorCallback callback = null;

	/** rotation group around x-axis */
	private TransformGroup rotateXGroup = null;

	/** rotation group around y-axis */
	private TransformGroup rotateYGroup = null;

	/** rotation around x-axis */
	private Transform3D rotateX = new Transform3D();

	/** rotation around y-axis */
	private Transform3D rotateY = new Transform3D();


	/**
	 * Creates a zoom behavior given the transform group.
	 * 
	 * @param transformGroup The transformGroup to operate on.
	 */
	public ZoomMouseBehavior(final TransformGroup transformGroup)
	{
		super(transformGroup);
	}

	/**
	 * Creates a default mouse zoom behavior.
	 */
	public ZoomMouseBehavior()
	{
		super(0);
	}

	/**
	 * Creates a zoom behavior. Note that this behavior still needs a transform
	 * group to work on (use setTransformGroup(tg)) and the transform group must
	 * add this behavior.
	 * 
	 * @param flags flags
	 */
	public ZoomMouseBehavior(final int flags)
	{
		super(flags);
	}

	/**
	 * Creates a zoom behavior that uses AWT listeners and behavior posts rather
	 * than WakeupOnAWTEvent. The behavior is added to the specified Component.
	 * A null component can be passed to specify the behavior should use
	 * listeners. Components can then be added to the behavior with the
	 * addListener(Component c) method.
	 * 
	 * @param c The Component to add the MouseListener and MouseMotionListener
	 *        to.
	 * @since Java 3D 1.2.1
	 */
	public ZoomMouseBehavior(final Component c)
	{
		super(c, 0);
	}

	/**
	 * Creates a zoom behavior that uses AWT listeners and behavior posts rather
	 * than WakeupOnAWTEvent. The behaviors is added to the specified Component
	 * and works on the given TransformGroup.
	 * 
	 * @param c The Component to add the MouseListener and MouseMotionListener
	 *        to. A null component can be passed to specify the behavior should
	 *        use listeners. Components can then be added to the behavior with
	 *        the addListener(Component c) method.
	 * @param transformGroup The TransformGroup to operate on.
	 * @since Java 3D 1.2.1
	 */
	public ZoomMouseBehavior(final Component c,
			final TransformGroup transformGroup)
	{
		super(c, transformGroup);
	}

	/**
	 * Creates a zoom behavior that uses AWT listeners and behavior posts rather
	 * than WakeupOnAWTEvent. The behavior is added to the specified Component.
	 * A null component can be passed to specify the behavior should use
	 * listeners. Components can then be added to the behavior with the
	 * addListener(Component c) method. Note that this behavior still needs a
	 * transform group to work on (use setTransformGroup(tg)) and the transform
	 * group must add this behavior.
	 * 
	 * @param c The Component to add the MouseListener and MouseMotionListener
	 *        to. A null component can be passed to specify the behavior should
	 *        use listeners. Components can then be added to the behavior with
	 *        the addListener(Component c) method.
	 * @param flags interesting flags (wakeup conditions).
	 * @since Java 3D 1.2.1
	 */
	public ZoomMouseBehavior(final Component c, final int flags)
	{
		super(c, flags);
	}

	/**
	 * @see javax.media.j3d.Behavior#initialize()
	 */
	public void initialize()
	{
		super.initialize();
		if ((this.flags & INVERT_INPUT) == INVERT_INPUT)
		{
			this.zFactor *= -1;
			this.invert = true;
		}
	}

	/**
	 * @return the y-axis movement multipler.
	 */
	public double getFactor()
	{
		return this.zFactor;
	}

	/**
	 * Set the y-axis movement multipler with factor.
	 * 
	 * @param factor the zoom factor
	 */
	public void setFactor(final double factor)
	{
		this.zFactor = factor;
	}

	/**
	 * Set the rotation around the x-axis.
	 * 
	 * @param rotateXGroup rotation around x-axis.
	 */
	public void setRotateXGroup(final TransformGroup rotateXGroup)
	{
		this.rotateXGroup = rotateXGroup;
	}

	/**
	 * Set the rotation around the y-axis.
	 * 
	 * @param rotateYGroup rotation around y-axis.
	 */
	public void setRotateYGroup(final TransformGroup rotateYGroup)
	{
		this.rotateYGroup = rotateYGroup;
	}

	/**
	 * @see javax.media.j3d.Behavior#processStimulus(java.util.Enumeration)
	 */
	public void processStimulus(final Enumeration criteria)
	{
		WakeupCriterion wakeup;
		AWTEvent[] events;
		MouseEvent evt;
		// int id;
		// int dx, dy;

		while (criteria.hasMoreElements())
		{
			wakeup = (WakeupCriterion) criteria.nextElement();
			if (wakeup instanceof WakeupOnAWTEvent)
			{
				events = ((WakeupOnAWTEvent) wakeup).getAWTEvent();
				if (events.length > 0)
				{
					evt = (MouseEvent) events[events.length - 1];
					doProcess(evt);
				}
			} else if (wakeup instanceof WakeupOnBehaviorPost)
			{
				while (true)
				{
					synchronized (this.mouseq)
					{
						if (this.mouseq.isEmpty())
						{
							break;
						}
						evt = (MouseEvent) this.mouseq.remove(0);
						// consolodate MOUSE_DRAG events
						while ((evt.getID() == MouseEvent.MOUSE_DRAGGED)
								&& !this.mouseq.isEmpty()
								&& (((MouseEvent) this.mouseq.get(0)).getID() == MouseEvent.MOUSE_DRAGGED))
						{
							evt = (MouseEvent) this.mouseq.remove(0);
						}
					}
					doProcess(evt);
				}
			}

		}
		wakeupOn(this.mouseCriterion);
	}

	/**
	 * @param evt mouse event
	 */
	void doProcess(final MouseEvent evt)
	{
		int id;
		int dy;

		processMouseEvent(evt);

		if (((this.buttonPress) && ((this.flags & MANUAL_WAKEUP) == 0))
				|| ((this.wakeUp) && ((this.flags & MANUAL_WAKEUP) != 0)))
		{
			id = evt.getID();
			if ((id == MouseEvent.MOUSE_DRAGGED) && evt.isAltDown()
					&& !evt.isMetaDown())
			{

				this.y = evt.getY();

				dy = this.y - this.y_last;

				if ((!this.reset) && ((Math.abs(dy) < 50)))
				{
					this.transformGroup.getTransform(this.currXform);

					this.translation.x = 0;
					this.translation.y = 0;
					this.translation.z = -dy * this.zFactor;

					this.transformX.setZero();

					// Now do some rotations around the x- and y-axis if these
					// are available.
					if ((this.rotateYGroup != null)
							&& (this.rotateXGroup != null))
					{
						this.rotateYGroup.getTransform(this.rotateY);
						this.rotateXGroup.getTransform(this.rotateX);
						this.transformX.set(this.rotateY);
						this.transformX.mul(this.rotateX);

						// transform the translation
						this.transformX.transform(this.translation);
					}
					this.transformX.set(this.translation);

					if (this.invert)
					{
						this.currXform.mul(this.currXform, this.transformX);
					} else
					{
						this.currXform.mul(this.transformX, this.currXform);
					}

					this.transformGroup.setTransform(this.currXform);

					this.transformChanged(this.currXform);

					if (this.callback != null)
					{
						this.callback.transformChanged(
								MouseBehaviorCallback.ZOOM, this.currXform);
					}

				} else
				{
					this.reset = false;
				}

				this.x_last = this.x;
				this.y_last = this.y;
			} else if (id == MouseEvent.MOUSE_PRESSED)
			{
				this.x_last = evt.getX();
				this.y_last = evt.getY();
			}
		}
	}

	/**
	 * Users can overload this method which is called every time the Behavior
	 * updates the transform
	 * 
	 * Default implementation does nothing
	 * 
	 * @param transform transform
	 */
	public void transformChanged(final Transform3D transform)
	{
		// nothing
	}

	/**
	 * The transformChanged method in the callback class will be called every
	 * time the transform is updated
	 * 
	 * @param callback callback
	 */
	public void setupCallback(final MouseBehaviorCallback callback)
	{
		this.callback = callback;
	}
}
