/*
 * @(#) Editor2DPanel.java 20-jul-2004 Copyright (c) 2002-2005 Delft University
 * of Technology Jaffalaan 5, 2628 BX Delft, the Netherlands. All rights
 * reserved. This software is proprietary information of Delft University of
 * Technology The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.editor2D;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.lang.reflect.Constructor;

import javax.naming.event.NamingEvent;

import nl.tudelft.simulation.dsol.animation.Editable;
import nl.tudelft.simulation.dsol.animation.D2.EditableRenderable2DInterface;
import nl.tudelft.simulation.dsol.animation.D2.Renderable2DInterface;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.animation2D.AnimationPanel;
import nl.tudelft.simulation.dsol.gui.animation2D.mouse.InputListener;
import nl.tudelft.simulation.dsol.gui.editor2D.actions.EditorUtilities;
import nl.tudelft.simulation.dsol.gui.editor2D.mouse.EditorInputListener;
import nl.tudelft.simulation.language.d3.CartesianPoint;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.language.reflection.ClassUtil;
import nl.tudelft.simulation.logger.Logger;

/**
 * Editor2DPanel.java <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision$ $Date$
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/royc/index.htm">Roy Chin
 *         </a>
 */
public class Editor2DPanel extends AnimationPanel
{
	/** Idle Mode */
	public static final int MODE_IDLE = 0;

	/** Selection Mode */
	public static final int MODE_SELECT = 1;

	/** New Object mMode */
	public static final int MODE_NEW = 2;

	/** Edit Object Mode */
	public static final int MODE_EDIT = 3;

	/** Delete Object Mode */
	public static final int MODE_DELETE = 4;

	/** Edit mode is 'move' */
	public static final int EDIT_MODE_MOVE = 1;

	/** Edit mode is 'rotate' */
	public static final int EDIT_MODE_ROTATE = 2;

	/** Placeholder size */
	public static final int PLACEHOLDERSIZE = 2;

	/** Line size of the representation of the local axis system */
	public static final int LOCALAXISSIZE = 20;

	/** Edit menu */
	private EditMenu editMenu = null;

	// EDITOR STATUS INFORMATION

	/** The selected mode */
	private int selectedMode = Editor2DPanel.MODE_IDLE;

	/** The selected edit mode */
	private int selectedEditMode = Editor2DPanel.EDIT_MODE_MOVE;

	/** The selected editable */
	private EditableRenderable2DInterface selectedEditableRenderable = null;

	/** Selected renderable class */
	private Class selectedRenderableClass = null;

	/** Selected control point */
	private CartesianPoint selectedPoint = null;

	/**
	 * Constructor
	 * 
	 * @param extent Extend
	 * @param size Size
	 * @param application DSOL Application
	 */
	public Editor2DPanel(final Rectangle2D extent, final Dimension size,
			final DSOLApplicationInterface application)
	{
		super(extent, size, application);

		// Remove all existing mouse interaction because
		// we're about to define our own.
		MouseListener[] mouseListeners = this.getMouseListeners();
		for (int i = 0; i < mouseListeners.length; i++)
		{
			this.removeMouseListener(mouseListeners[i]);
		}
		MouseMotionListener[] mouseMotionListeners = this
				.getMouseMotionListeners();
		for (int i = 0; i < mouseMotionListeners.length; i++)
		{
			this.removeMouseMotionListener(mouseMotionListeners[i]);
		}

		// If a custom mouse editor is provided then use that one
		// otherwise use the default one.
		String mouseEditorClassName = application.getProperties().getProperty(
				"dsol-gui.editor2D.panel.editorInputListener");
		if (mouseEditorClassName != null)
		{
			try
			{
				Class editorClass = Class.forName(mouseEditorClassName);
				Constructor constructor = ClassUtil.resolveConstructor(
						editorClass, new Class[]{
								DSOLApplicationInterface.class,
								Editor2DPanel.class});
				InputListener mouseListener = (InputListener) constructor
						.newInstance(new Object[]{application, this});
				this.addMouseListener(mouseListener);
				this.addMouseMotionListener(mouseListener);
				this.addMouseWheelListener(mouseListener);
			} catch (Exception exception)
			{
				Logger.warning(this, "<init>", "could not load "
						+ mouseEditorClassName);
				// add our mouse editor
				InputListener mouse = new EditorInputListener(application, this);
				this.addMouseListener(mouse);
				this.addMouseMotionListener(mouse);
				this.addMouseWheelListener(mouse);
			}
		} else
		{
			// add our mouse editor
			InputListener mouse = new EditorInputListener(application, this);
			this.addMouseListener(mouse);
			this.addMouseMotionListener(mouse);
			this.addMouseWheelListener(mouse);
		}

		// The edit menu
		this.editMenu = new EditMenu("Edit", this);

	}

	/**
	 * @see javax.naming.event.NamespaceChangeListener#objectAdded(javax.naming.event.NamingEvent)
	 */
	public void objectAdded(final NamingEvent namingEvent)
	{
		super.objectAdded(namingEvent);
		// Select the last added object
		Renderable2DInterface element = (Renderable2DInterface) namingEvent
				.getNewBinding().getObject();
		if (element instanceof EditableRenderable2DInterface)
		{
			this
					.setSelectedEditableRenderable((EditableRenderable2DInterface) element);
		}
	}

	/**
	 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
	 */
	public void paintComponent(final Graphics g)
	{
		super.paintComponent(g);

		// Draw outline of selected
		if (this.getSelectedEditableRenderable() != null)
		{
			try
			{
				g.setColor(Color.RED);
				EditableRenderable2DInterface selected = this
						.getSelectedEditableRenderable();
				Editable editable = (Editable) selected.getSource();
				CartesianPoint[] vertices = editable.getVertices();
				CartesianPoint[] positions = new CartesianPoint[vertices.length];
				for (int i = 0; i < positions.length; i++)
				{
					positions[i] = EditorUtilities.convertToGlobalCoordinates(
							vertices[i], editable.getLocation());
				}
				// Draw line segments
				for (int i = 0; i < positions.length - 1; i++)
				{
					Point2D p1 = Renderable2DInterface.Util
							.getScreenCoordinates(new Point2D.Double(
									positions[i].x, positions[i].y),
									this.extent, this.getSize());
					Point2D p2 = Renderable2DInterface.Util
							.getScreenCoordinates(new Point2D.Double(
									positions[i + 1].x, positions[i + 1].y),
									this.extent, this.getSize());

					g.drawLine((int) p1.getX(), (int) p1.getY(), (int) p2
							.getX(), (int) p2.getY());

					if (positions[i] == this.getSelectedPoint())
					{
						g.setColor(Color.YELLOW);
					}
					g.fillRect((int) p1.getX() - Editor2DPanel.PLACEHOLDERSIZE,
							(int) p1.getY() - Editor2DPanel.PLACEHOLDERSIZE,
							Editor2DPanel.PLACEHOLDERSIZE * 2 + 1,
							Editor2DPanel.PLACEHOLDERSIZE * 2 + 1);
					g.setColor(Color.RED);
				}

				Point2D p1 = Renderable2DInterface.Util.getScreenCoordinates(
						new Point2D.Double(positions[0].x, positions[0].y),
						this.extent, this.getSize());
				Point2D p2 = Renderable2DInterface.Util.getScreenCoordinates(
						new Point2D.Double(positions[positions.length - 1].x,
								positions[positions.length - 1].y),
						this.extent, this.getSize());
				// Draw closing line
				if (selected.isClosedShape())
				{
					g.drawLine((int) p1.getX(), (int) p1.getY(), (int) p2
							.getX(), (int) p2.getY());
				}
				if (positions[positions.length - 1] == this.getSelectedPoint())
				{
					g.setColor(Color.YELLOW);
				}
				g.fillRect((int) p2.getX() - Editor2DPanel.PLACEHOLDERSIZE,
						(int) p2.getY() - Editor2DPanel.PLACEHOLDERSIZE,
						Editor2DPanel.PLACEHOLDERSIZE * 2 + 1,
						Editor2DPanel.PLACEHOLDERSIZE * 2 + 1);
				g.setColor(Color.RED);

				// Draw a cross at the Location of the object
				DirectedPoint location = ((Renderable2DInterface) selected)
						.getSource().getLocation();
				Point2D loc = Renderable2DInterface.Util.getScreenCoordinates(
						location.to2D(), this.extent, this.getSize());
				g.drawLine((int) (loc.getX() - 2), (int) (loc.getY()),
						(int) (loc.getX() + 2), (int) (loc.getY()));
				g.drawLine((int) (loc.getX()), (int) (loc.getY() - 2),
						(int) (loc.getX()), (int) (loc.getY() + 2));
				// Draw local axis system
				g.drawLine((int) (loc.getX()), (int) (loc.getY()), (int) (loc
						.getX() + Math.cos(location.getRotZ())
						* Editor2DPanel.LOCALAXISSIZE),
						(int) (loc.getY() - Math.sin(location.getRotZ())
								* Editor2DPanel.LOCALAXISSIZE));
				g.drawLine((int) (loc.getX()), (int) (loc.getY()), (int) (loc
						.getX() - Math.sin(location.getRotZ())
						* Editor2DPanel.LOCALAXISSIZE),
						(int) (loc.getY() - Math.cos(location.getRotZ())
								* Editor2DPanel.LOCALAXISSIZE));
				g.drawString("x", (int) (loc.getX() + Math.cos(location
						.getRotZ())
						* Editor2DPanel.LOCALAXISSIZE),
						(int) (loc.getY() - Math.sin(location.getRotZ())
								* Editor2DPanel.LOCALAXISSIZE));
				g.drawString("y", (int) (loc.getX() - Math.sin(location
						.getRotZ())
						* Editor2DPanel.LOCALAXISSIZE),
						(int) (loc.getY() - Math.cos(location.getRotZ())
								* Editor2DPanel.LOCALAXISSIZE));
			} catch (Exception exception)
			{
				Logger.warning(this, "paintComponent", exception);
			}
		}

	}

	/**
	 * @return returns the editMenu.
	 */
	public EditMenu getEditMenu()
	{
		return this.editMenu;
	}

	// EDITOR STATUS
	/**
	 * @return returns the selectedMode.
	 */
	public int getSelectedMode()
	{
		return this.selectedMode;
	}

	/**
	 * @param mode the selectedMode to set.
	 */
	public void setSelectedMode(final int mode)
	{
		this.selectedMode = mode;
	}

	/**
	 * @return returns the selected editable renderable.
	 */
	public EditableRenderable2DInterface getSelectedEditableRenderable()
	{
		return this.selectedEditableRenderable;
	}

	/**
	 * @param selectedEditableRenderable The selected editable renderable to
	 *        set.
	 */
	public void setSelectedEditableRenderable(
			final EditableRenderable2DInterface selectedEditableRenderable)
	{
		this.selectedEditableRenderable = selectedEditableRenderable;
	}

	/**
	 * @return returns the selectedRenderableClass.
	 */
	public Class getSelectedRenderableClass()
	{
		return this.selectedRenderableClass;
	}

	/**
	 * @param selectedRenderableClass The selectedRenderableClass to set.
	 */
	public void setSelectedRenderableClass(final Class selectedRenderableClass)
	{
		this.selectedRenderableClass = selectedRenderableClass;
	}

	/**
	 * @return returns the selectedEditMode.
	 */
	public int getSelectedEditMode()
	{
		return this.selectedEditMode;
	}

	/**
	 * @param selectedEditMode The selectedEditMode to set.
	 */
	public void setSelectedEditMode(final int selectedEditMode)
	{
		this.selectedEditMode = selectedEditMode;
	}

	/**
	 * @return returns the selectedPoint.
	 */
	public CartesianPoint getSelectedPoint()
	{
		return this.selectedPoint;
	}

	/**
	 * @param selectedPoint The selectedPoint to set.
	 */
	public void setSelectedPoint(final CartesianPoint selectedPoint)
	{
		this.selectedPoint = selectedPoint;
	}
}