/*
 * @(#) EditorUtilities.java 28-jul-2004 Copyright (c) 2002-2005 Delft
 * University of Technology Jaffalaan 5, 2628 BX Delft, the Netherlands. All
 * rights reserved. This software is proprietary information of Delft University
 * of Technology The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.editor2D.actions;

import java.awt.geom.Point2D;
import java.lang.reflect.Constructor;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import nl.tudelft.simulation.dsol.animation.Editable;
import nl.tudelft.simulation.dsol.animation.D2.EditableRenderable2DInterface;
import nl.tudelft.simulation.dsol.animation.D2.Renderable2DInterface;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.editor2D.Editor2DPanel;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.language.d3.CartesianPoint;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.language.reflection.ClassUtil;
import nl.tudelft.simulation.logger.Logger;

/**
 * Utility methods to select, create, edit and delete editables
 * <p>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision$ $Date$
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/royc/index.htm">Roy Chin
 *         </a>
 */
public final class EditorUtilities
{

	/** the default offset value in x and y dirextion for new points */
	public static final int DEFAULT_OFFSET = 8;

	/** a hidden constructor */
	private EditorUtilities()
	{
		// Do nothing
	}

	/**
	 * determine the selected object
	 * 
	 * @param worldCoordinate point in world coordinates
	 * @param panel the editable animation panel
	 * @return the selected renderable
	 */
	public static Renderable2DInterface selectEditable(
			final Point2D worldCoordinate, final Editor2DPanel panel)
	{
		// Determine which object was selected.
		Renderable2DInterface selected = EditorUtilities.determineSelected(
				worldCoordinate, panel);
		if (selected == null)
		{
			panel.setSelectedEditableRenderable(null);
		} else
		{
			try
			{
				if (selected instanceof EditableRenderable2DInterface)
				{
					panel
							.setSelectedEditableRenderable((EditableRenderable2DInterface) selected);
				}
			} catch (ClassCastException exception)
			{
				Logger.warning(EditorUtilities.class, "selectEditable",
						exception);
			}
		}
		return selected;
	}

	/**
	 * instantiate a new editable
	 * 
	 * @param worldCoordinate point in world coordinates
	 * @param application the application
	 * @param panel the editable animation panel
	 */
	public static void instantiateNewEditable(final Point2D worldCoordinate,
			final DSOLApplicationInterface application,
			final Editor2DPanel panel)
	{
		panel.setSelectedEditableRenderable(null);
		Class editableClass = panel.getSelectedRenderableClass();

		// Now try to instantiate
		try
		{
			// Create a new editable
			Class[] argsClass = new Class[]{SimulatorInterface.class,
					DirectedPoint.class};
			Object[] args = new Object[]{
					application.getExperiment().getSimulator(),
					new DirectedPoint(worldCoordinate.getX(), worldCoordinate
							.getY(), 0)};
			Constructor constructor = ClassUtil.resolveConstructor(
					editableClass, argsClass);
			constructor.newInstance(args);
		} catch (Exception exception)
		{
			Logger.warning(EditorUtilities.class, "instantiateNewEditable",
					exception);
		}
	}

	/**
	 * move the selected control point
	 * 
	 * @param target the editableRenderable which is moved.
	 * @param worldCoordinate point in world coordinates where to move to
	 * @param panel the Editor2D panel
	 */
	public static void moveSelectedPoint(
			final EditableRenderable2DInterface target,
			final Point2D worldCoordinate, final Editor2DPanel panel)
	{
		try
		{
			Editable editable = (Editable) target.getSource();
			CartesianPoint[] vertices = editable.getVertices();

			if (target.allowEditPoints())
			{
				// Determine the new CartesianPoints
				CartesianPoint[] newPos = new CartesianPoint[vertices.length];
				for (int i = 0; i < vertices.length; i++)
				{
					newPos[i] = new CartesianPoint();
					if (vertices[i] == panel.getSelectedPoint())
					{
						CartesianPoint coord = EditorUtilities
								.convertToLocalCoordinates(new CartesianPoint(
										worldCoordinate.getX(), worldCoordinate
												.getY(), 0), editable
										.getLocation());

						newPos[i].x = coord.x;
						newPos[i].y = coord.y;
						newPos[i].z = vertices[i].z;
					} else
					{
						newPos[i].x = vertices[i].x;
						newPos[i].y = vertices[i].y;
						newPos[i].z = vertices[i].z;
					}
				}
				// Now let the selected editable decide for itself
				// how it handles this modification.
				editable.setVertices(newPos);
			}
		} catch (RemoteException exception)
		{
			Logger.warning(EditorUtilities.class, "moveSelectedPoint",
					exception);
		}
	}

	/**
	 * move the object
	 * 
	 * @param target the target editable renderable
	 * @param newCoordinate Point in world coordinates where to move to
	 * @param oldCoordinate Point in world coordinates where the mouse was the
	 *        last iteration, because we move relative to this coordinate
	 */
	public static void moveEditable(final EditableRenderable2DInterface target,
			final Point2D newCoordinate, final Point2D oldCoordinate)
	{
		try
		{
			Editable editable = (Editable) target.getSource();
			if (editable != null)
			{
				if (target.allowMove())
				{
					double dx = newCoordinate.getX() - oldCoordinate.getX();
					double dy = newCoordinate.getY() - oldCoordinate.getY();
					DirectedPoint location = editable.getLocation();
					location.x += dx;
					location.y += dy;
				}
			}
		} catch (RemoteException exception)
		{
			Logger.warning(EditorUtilities.class, "moveEditable", exception);
		}
	}

	/**
	 * rotate the editable
	 * 
	 * @param target the target editable renderable
	 * @param newCoordinate point in world coordinates where to rotate to
	 * @param oldCoordinate Point in world coordinates where the mouse was the
	 *        last iteration, because we move relative to this coordinate
	 * @param centerCoordinate the center of rotation in world coordinates
	 */
	public static void rotateEditable(
			final EditableRenderable2DInterface target,
			final Point2D newCoordinate, final Point2D centerCoordinate,
			final Point2D oldCoordinate)
	{
		try
		{
			Editable editable = (Editable) target.getSource();
			if (editable != null)
			{
				if (target.allowRotate())
				{
					double dx = newCoordinate.getX() - oldCoordinate.getX();
					double dy = newCoordinate.getY() - oldCoordinate.getY();
					double dxR = oldCoordinate.getX() - centerCoordinate.getX();
					double dyR = oldCoordinate.getY() - centerCoordinate.getY();
					double dxR2 = newCoordinate.getX()
							- centerCoordinate.getX();
					double dyR2 = newCoordinate.getY()
							- centerCoordinate.getY();
					double distY = Math.sqrt(dx * dx + dy * dy);
					double distX = Math.sqrt(dxR * dxR + dyR * dyR);
					int factor = 1;
					if (Math.atan2(dyR, dxR) > Math.atan2(dyR2, dxR2))
					{
						factor = -1;
					}

					double angle = factor * Math.atan2(distY, distX);

					// Change the location too
					DirectedPoint location = editable.getLocation();
					double dxP = location.x - centerCoordinate.getX();
					double dyP = location.y - centerCoordinate.getY();
					double distanceP = Math.sqrt(dxP * dxP + dyP * dyP);
					double angleP = Math.atan2(dyP, dxP);
					location.x = centerCoordinate.getX() + distanceP
							* Math.cos(angle + angleP);
					location.y = centerCoordinate.getY() + distanceP
							* Math.sin(angle + angleP);
					location.setRotZ(angle + location.getRotZ());
				}
			}
		} catch (RemoteException exception)
		{
			Logger.warning(EditorUtilities.class, "rotateEditable", exception);
		}
	}

	/**
	 * delete the selected editable
	 * 
	 * @param target the target editable renderable
	 * @param panel the editable animation panel
	 */
	public static void deleteEditable(
			final EditableRenderable2DInterface target,
			final Editor2DPanel panel)
	{
		Editable editable = (Editable) target.getSource();
		if (editable != null)
		{
			try
			{
				if (target.allowDelete())
				{
					target.destroy();
					panel.setSelectedEditableRenderable(null);
					panel.setSelectedPoint(null);
					panel.repaint();
				}
			} catch (RemoteException exception)
			{
				Logger.warning(EditorUtilities.class, "deleteEditable",
						exception);
			}
		}
	}

	/**
	 * add a point behind the last point of the editable
	 * 
	 * @param target the target editable renderable
	 * @param panel the animation panel
	 */
	public static void addPointToEditable(
			final EditableRenderable2DInterface target,
			final Editor2DPanel panel)
	{
		try
		{
			// The distance in screen coordinates that the new point
			// will be offset in x and y direction from the last point.
			double delta = EditorUtilities.DEFAULT_OFFSET
					* Renderable2DInterface.Util.getScale(panel.getExtent(),
							panel.getSize());

			Editable editable = (Editable) target.getSource();
			CartesianPoint[] vertices = editable.getVertices();

			// Add the point
			if ((target.allowAddOrDeletePoints())
					&& (vertices.length < target.getMaxNumberOfPoints()))
			{
				CartesianPoint[] newPos = new CartesianPoint[vertices.length + 1];
				for (int i = 0; i < newPos.length; i++)
				{
					newPos[i] = new CartesianPoint();
					if (i == (newPos.length - 1))
					{
						newPos[i].x = vertices[i - 1].x + delta;
						newPos[i].y = vertices[i - 1].y + delta;
						newPos[i].z = vertices[i - 1].z;
					} else
					{
						newPos[i].x = vertices[i].x;
						newPos[i].y = vertices[i].y;
						newPos[i].z = vertices[i].z;
					}
				}
				editable.setVertices(newPos);
				panel.repaint();
			}
		} catch (RemoteException exception)
		{
			Logger.warning(EditorUtilities.class, "addPointToEditable",
					exception);
		}
	}

	/**
	 * determine selected object as the mouse CartesianPoint. This is the
	 * topmost target when objects overlap.
	 * 
	 * @param worldCoordinate the selected point in world coordinates
	 * @param panel the editable animation panel
	 * @return a renderable if there is one at the selected point
	 */
	public static Renderable2DInterface determineSelected(
			final Point2D worldCoordinate, final Editor2DPanel panel)
	{
		Renderable2DInterface selected = null;
		Renderable2DInterface previous = panel.getSelectedEditableRenderable();
		List targets = EditorUtilities.determineTargets(worldCoordinate, panel);

		if (previous != null)
		{
			// Check if 'previous' is in 'targets'
			int index = -1;
			for (int i = 0; i < targets.size(); i++)
			{
				if (previous == (Renderable2DInterface) targets.get(i))
				{
					index = i;
				}
			}
			if (index >= 0)
			{
				if (index + 1 < targets.size())
				{
					selected = (Renderable2DInterface) targets.get(index + 1);
				} else
				{
					selected = (Renderable2DInterface) targets.get(0);
				}
			} else if (targets.size() > 0)
			{
				selected = (Renderable2DInterface) targets.get(0);
			}
		} else
		{
			try
			{
				double zValue = -Double.MAX_VALUE;
				for (Iterator i = targets.iterator(); i.hasNext();)
				{
					Renderable2DInterface next = (Renderable2DInterface) i
							.next();

					double z = next.getSource().getLocation().z;
					if (z > zValue)
					{
						zValue = z;
						selected = next;
					}

				}
			} catch (RemoteException exception)
			{
				Logger.warning(EditorUtilities.class, "determineSelected",
						exception);
			}

		}

		return selected;
	}

	/**
	 * determine the targeted objects at the mouse CartesianPoint
	 * 
	 * @param worldCoordinate point in world coordinates
	 * @param panel the editable animation panel
	 * @return targeted objects
	 */
	public static List determineTargets(final Point2D worldCoordinate,
			final Editor2DPanel panel)
	{
		List targets = new ArrayList();
		try
		{
			for (Iterator i = panel.getElements().iterator(); i.hasNext();)
			{
				Renderable2DInterface renderable = (Renderable2DInterface) i
						.next();
				if (renderable.contains(worldCoordinate, panel.getExtent(),
						panel.getSize()))
				{
					targets.add(renderable);
				}
			}
		} catch (Exception exception)
		{
			Logger
					.warning(EditorUtilities.class, "determineTargets",
							exception);
		}
		return targets;
	}

	/**
	 * converts a coordinate in local coordinates to global coordinates note:
	 * this works only for 2D
	 * 
	 * @param point point in local coordinates
	 * @param location location vector
	 * @return point in global coordinates
	 */
	public static CartesianPoint convertToGlobalCoordinates(
			final CartesianPoint point, final DirectedPoint location)
	{
		CartesianPoint global = new CartesianPoint();
		// Rotate
		double angle = Math.atan2(point.y, point.x);
		double length = Math.sqrt(point.x * point.x + point.y * point.y);
		double rotation = location.getRotZ() + angle;
		global.x = length * Math.cos(rotation);
		global.y = length * Math.sin(rotation);

		// Translate
		global.x += location.x;
		global.y += location.y;
		return global;
	}

	/**
	 * converts a coordinate in global coordinates to local coordinates note:
	 * this works only for 2D
	 * 
	 * @param point point in local coordinates
	 * @param location location vector
	 * @return point in global coordinates
	 */
	public static CartesianPoint convertToLocalCoordinates(
			final CartesianPoint point, final DirectedPoint location)
	{
		CartesianPoint local = new CartesianPoint();

		// Translate
		local.x = point.x - location.x;
		local.y = point.y - location.y;

		// Rotate
		double angle = Math.atan2(local.y, local.x);
		double length = Math.sqrt(local.x * local.x + local.y * local.y);
		double rotation = angle - location.getRotZ();
		local.x = length * Math.cos(rotation);
		local.y = length * Math.sin(rotation);
		return local;
	}
}