/*
 * @(#) ControlPanel.java Oct 14, 2003
 * 
 * Copyright (c) 2002-2005 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands. All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.panels;

import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.rmi.RemoteException;
import java.text.NumberFormat;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JPanel;
import javax.swing.JTextField;

import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.experiment.TimeUnitInterface;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.actions.AnimationDelaySlider;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DESSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.logger.Logger;

/**
 * A ControlPanel <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision: 1.4 $ $Date: 2005/01/13 16:28:55 $
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja">Peter Jacobs </a>
 */
public class ControlPanel extends JPanel implements EventListenerInterface
{
	/** the timeStepSlider */
	// private TimeStepSlider timeStepSlider;
	/** the animationDelaySlider */
	private AnimationDelaySlider animationDelaySlider;

	/** the animationDelayTextField */
	private JTextField animationDelayTextField = null;

	/** the timeSteptextField */
	protected JTextField timeStepTextField = null;

	/** timeStepPanel */
	private JPanel timeStepPanel = new JPanel();

	/** the animationDelayFormatter */
	private NumberFormat animationDelayFormatter = NumberFormat.getInstance();

	/** the DSOLApplication */
	protected DSOLApplicationInterface application = null;

	/**
	 * constructs a new ControlPanel
	 * 
	 * @param application the application
	 */
	public ControlPanel(final DSOLApplicationInterface application)
	{
		super();
		this.application = application;
		this.animationDelayFormatter.setMinimumFractionDigits(0);
		this.animationDelayFormatter.setMaximumFractionDigits(0);
		this.initialize();
		try
		{
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
			this.notify(new Event(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT,
					this.application, this.application.getExperiment()));
		} catch (RemoteException exception)
		{
			Logger.warning(this, "ControlPanel", exception);
		}
	}

	/**
	 * initializes the console
	 */
	private void initialize()
	{
		this.setLayout(new BorderLayout());
		this.setBorder(BorderFactory.createEmptyBorder());

		JPanel content = new JPanel();
		content.setLayout(new BoxLayout(content, BoxLayout.Y_AXIS));
		content.setBorder(BorderFactory.createEmptyBorder());

		this.timeStepPanel.setLayout(new BorderLayout());

		String title = TimeUnitInterface.UNIT.toString();
		this.timeStepPanel.setBorder(BorderFactory
				.createTitledBorder("time step ( " + title
						+ " on simulation clock )"));

		JTextField timeSteplabel = new JTextField("time step:  ");
		this.timeStepPanel.add(timeSteplabel, BorderLayout.WEST);

		this.timeStepTextField = new JTextField();
		this.timeStepTextField.setEditable(true);
		Font font = this.timeStepTextField.getFont().deriveFont(Font.BOLD);
		this.timeStepTextField.setFont(font);
		this.timeStepTextField.addActionListener(new ActionListener()
		{
			public void actionPerformed(final ActionEvent e)
			{
				DESSSimulatorInterface simulator = (DESSSimulatorInterface) ControlPanel.this.application
						.getExperiment().getSimulator();
				try
				{
					double value = new Double(e.getActionCommand())
							.doubleValue();
					if (value <= 0)
					{
						throw new NumberFormatException("timeStep value <=0");
					}
					simulator.setTimeStep(value);
				} catch (Exception numberFormatException)
				{
					Logger.warning(this, "timeStepEditor",
							numberFormatException);

					double value = DESSSimulatorInterface.DEFAULT_TIME_STEP;
					try
					{
						value = simulator.getTimeStep();
					} catch (RemoteException e1)
					{
						Logger.warning(this, "nonsense", e1);
					}
					ControlPanel.this.timeStepTextField.setText(value + "");
				}
			}
		});
		this.timeStepTextField.setBorder(BorderFactory.createEmptyBorder());
		this.timeStepPanel.add(this.timeStepTextField, BorderLayout.CENTER);

		content.add(this.timeStepPanel);

		JPanel animationDelayPanel = new JPanel();
		animationDelayPanel.setLayout(new BorderLayout());
		animationDelayPanel
				.setBorder(BorderFactory
						.createTitledBorder("animation delay ( milliseconds on wall clock ) "));

		this.animationDelaySlider = new AnimationDelaySlider(this.application);
		this.animationDelaySlider.setName("animationDelaySlider");
		this.animationDelaySlider.setBorder(BorderFactory.createEmptyBorder(0,
				10, 0, 10));

		animationDelayPanel.add(this.animationDelaySlider, BorderLayout.CENTER);

		this.animationDelayTextField = new JTextField();
		this.animationDelayTextField.setEditable(false);
		this.animationDelayTextField.setBorder(BorderFactory
				.createEmptyBorder());
		animationDelayPanel
				.add(this.animationDelayTextField, BorderLayout.EAST);

		try
		{
			this.timeStepTextField
					.setText(DESSSimulatorInterface.DEFAULT_TIME_STEP + "");
			this.animationDelayTextField.setText(this.animationDelayFormatter
					.format(AnimatorInterface.DEFAULT_ANIMATION_DELAY));
			this.animationDelaySlider.setValue((int) Math.round((Math
					.log(AnimatorInterface.DEFAULT_ANIMATION_DELAY) / Math
					.log(10)) * 1000));
		} catch (Exception exception)
		{
			Logger.warning(this, "initialize", exception);
		}
		content.add(animationDelayPanel);
		this.add(content);
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event) throws RemoteException
	{
		if (event.getType().equals(
				DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT)
				&& event.getSource() instanceof DSOLApplicationInterface)
		{
			if (event.getContent() != null)
			{
				Experiment experiment = (Experiment) event.getContent();
				experiment
						.addListener(this, Experiment.SIMULATOR_CHANGED_EVENT);
				this.notify(new Event(Experiment.SIMULATOR_CHANGED_EVENT,
						experiment, experiment.getSimulator()));

			}
			return;
		}
		if (event.getType().equals(Experiment.SIMULATOR_CHANGED_EVENT)
				&& event.getSource() instanceof Experiment)
		{
			if (event.getContent() != null)
			{
				AnimatorInterface simulator = (AnimatorInterface) event
						.getContent();
				simulator.addListener(this,
						AnimatorInterface.ANIMATION_DELAY_CHANGED_EVENT);
				simulator.addListener(this,
						DESSSimulatorInterface.TIME_STEP_CHANGED_EVENT);
				simulator.addListener(this,
						SimulatorInterface.START_REPLICATION_EVENT);
				this.notify(new Event(
						AnimatorInterface.ANIMATION_DELAY_CHANGED_EVENT,
						simulator, new Long(simulator.getAnimationDelay())));
				this.notify(new Event(
						SimulatorInterface.START_REPLICATION_EVENT, simulator,
						null));
				this.notify(new Event(
						DESSSimulatorInterface.TIME_STEP_CHANGED_EVENT,
						simulator, new Double(simulator.getTimeStep())));
			}
			return;
		}
		if (event.getSource() instanceof SimulatorInterface
				&& event.getType().equals(
						SimulatorInterface.START_REPLICATION_EVENT))
		{
			SimulatorInterface simulator = (SimulatorInterface) event
					.getSource();
			if (simulator.getReplication() != null)
			{
				this.timeStepPanel
						.setBorder(BorderFactory
								.createTitledBorder("time step ( "
										+ simulator.getReplication()
												.getRunControl().getTreatment()
												.getTimeUnit().toString()
										+ " on simulation clock )"));
				this.validate();
				this.repaint();
				return;
			}
		}
		if (event.getSource() instanceof AnimatorInterface)
		{
			if (event.getType().equals(
					AnimatorInterface.ANIMATION_DELAY_CHANGED_EVENT))
			{
				this.animationDelayTextField
						.setText(this.animationDelayFormatter
								.format(((Long) event.getContent()).longValue()));
			}
			if (event.getType().equals(
					DESSSimulatorInterface.TIME_STEP_CHANGED_EVENT))
			{
				this.timeStepTextField.setText(event.getContent().toString());
			}
			this.validate();
			this.repaint();
			return;
		}
	}
}