/*
 * @(#) Statusbar.java Oct 13, 2003
 * 
 * Copyright (c) 2002-2005 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands. All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.panels;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.rmi.RemoteException;
import java.text.NumberFormat;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.UIManager;

import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.actions.FastForwardAction;
import nl.tudelft.simulation.dsol.gui.actions.ResetAction;
import nl.tudelft.simulation.dsol.gui.actions.StartAction;
import nl.tudelft.simulation.dsol.gui.actions.StepAction;
import nl.tudelft.simulation.dsol.gui.actions.StopAction;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.logger.Logger;

/**
 * The StatusBar <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision: 1.4 $ $Date: 2005/01/13 16:28:54 $
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja">Peter Jacobs </a>
 */
public class Statusbar extends JPanel implements EventListenerInterface
{

	/** the timeField of the statusBar */
	private JTextField timeField = new JTextField();

	/** the replicationField of the statusBar */
	private JTextField replicationField = new JTextField();

	/** the replicationField of the statusBar */
	private JTextField treatmentField = new JTextField();

	/** the parent DSOL Panel */
	private DSOLApplicationInterface application = null;

	/** the units */
	private String units = null;

	/** the simulatorTime */
	private double simulatorTime = Double.NaN;

	/**
	 * constructs a new Statusbar
	 * 
	 * @param application the application
	 */
	public Statusbar(final DSOLApplicationInterface application)
	{
		super();
		this.application = application;
		this.initialize();
		try
		{
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
			this.notify(new Event(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT,
					this.application, this.application.getExperiment()));
			if (this.application.getExperiment() != null
					&& this.application.getExperiment().getSimulator() != null)
			{
				if (this.application.getExperiment().getSimulator()
						.getReplication() != null)
				{
					this.notify(new Event(
							SimulatorInterface.START_REPLICATION_EVENT,
							this.application.getExperiment().getSimulator(),
							this.application.getExperiment().getSimulator()
									.getReplication()));
				}
			}
		} catch (Exception exception)
		{
			Logger.warning(this, "Statusbar", exception);
		}
	}

	/**
	 * initializes the statusbar
	 */
	private void initialize()
	{
		Color backgroundColor = UIManager.getColor("MenuBar.background");
		this.setBackground(backgroundColor);
		this.setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
		JPanel controlPanel = new JPanel();
		controlPanel.setBackground(backgroundColor);
		controlPanel.setLayout(new BoxLayout(controlPanel, BoxLayout.X_AXIS));
		StepAction stepAction = new StepAction(this.application);
		JButton stepButton = new JButton(stepAction);
		stepButton.setText("");
		stepButton.setBackground(backgroundColor);
		stepButton.setToolTipText("steps the simulator");
		stepButton.setBorder(BorderFactory.createEmptyBorder());
		controlPanel.add(stepButton);
		StartAction startAction = new StartAction(this.application);
		JButton startButton = new JButton(startAction);
		startButton.setText("");
		startButton.setBackground(backgroundColor);
		startButton.setToolTipText("starts the simulator");
		startButton.setBorder(BorderFactory.createEmptyBorder());
		controlPanel.add(startButton);
		StopAction stopAction = new StopAction(this.application);
		JButton stopButton = new JButton(stopAction);
		stopButton.setText("");
		stopButton.setBackground(backgroundColor);
		stopButton.setToolTipText("stops the simulator");
		stopButton.setBorder(BorderFactory.createEmptyBorder());
		controlPanel.add(stopButton);
		FastForwardAction fastForwardAction = new FastForwardAction(
				this.application);
		JButton fastForwardButton = new JButton(fastForwardAction);
		fastForwardButton.setText("");
		fastForwardButton.setBackground(backgroundColor);
		fastForwardButton.setToolTipText("fast forwards the simulator");
		fastForwardButton.setBorder(BorderFactory.createEmptyBorder());
		controlPanel.add(fastForwardButton);
		ResetAction resetAction = new ResetAction(this.application);
		JButton resetButton = new JButton(resetAction);
		resetButton.setText("");
		resetButton.setBackground(backgroundColor);
		resetButton.setToolTipText("resets the experiment");
		resetButton.setBorder(BorderFactory.createEmptyBorder());
		controlPanel.add(resetButton);
		JPanel statusPanel = new JPanel();
		statusPanel.setBackground(backgroundColor);
		statusPanel.setBorder(BorderFactory.createEmptyBorder());
		statusPanel.setLayout(new BorderLayout());
		this.timeField.setBackground(Color.WHITE);
		this.timeField.setPreferredSize(new Dimension(200, 12));
		this.timeField.setEditable(false);
		this.timeField.setText("Time:");
		this.timeField.setToolTipText("The simulator time");
		statusPanel.add(this.timeField, BorderLayout.WEST);
		JPanel eastSide = new JPanel();
		eastSide.setBackground(backgroundColor);
		eastSide.setBorder(BorderFactory.createEmptyBorder());
		eastSide.setLayout(new BorderLayout());
		this.replicationField.setBackground(Color.WHITE);
		this.replicationField.setEditable(false);
		this.replicationField.setText("Rep(-/-)");
		this.replicationField
				.setToolTipText("The currently running replication");
		this.treatmentField.setBackground(Color.WHITE);
		this.treatmentField.setEditable(false);
		this.treatmentField.setText("Treat(-/-)");
		this.treatmentField.setToolTipText("The currently running treatment");
		eastSide.add(this.replicationField, BorderLayout.WEST);
		eastSide.add(this.treatmentField, BorderLayout.EAST);
		statusPanel.add(eastSide, BorderLayout.EAST);
		this.add(controlPanel);
		this.add(statusPanel);
		new TimeUpdater(this).start();
	}

	/**
	 * gets simulatorTime
	 * 
	 * @return Returns the simulatorTime.
	 */
	protected double getSimulatorTime()
	{
		return this.simulatorTime;
	}

	/**
	 * gets timeField
	 * 
	 * @return Returns the timeField.
	 */
	protected JTextField getTimeField()
	{
		return this.timeField;
	}

	/**
	 * gets units
	 * 
	 * @return Returns the units.
	 */
	protected String getUnits()
	{
		return this.units;
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event) throws RemoteException
	{
		if (event.getSource() instanceof DSOLApplicationInterface)
		{
			if (event.getType().equals(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT))
			{
				if (event.getContent() != null)
				{
					Experiment experiment = (Experiment) event.getContent();
					experiment.addListener(this,
							Experiment.SIMULATOR_CHANGED_EVENT);
					this.notify(new Event(Experiment.SIMULATOR_CHANGED_EVENT,
							experiment, experiment.getSimulator()));
				}
				return;
			}
		} else if (event.getSource() instanceof Experiment)
		{
			if (event.getType().equals(Experiment.SIMULATOR_CHANGED_EVENT))
			{
				this.simulatorTime = Double.NaN;
				if (event.getContent() != null)
				{
					SimulatorInterface simulator = (SimulatorInterface) event
							.getContent();
					simulator.addListener(this,
							SimulatorInterface.START_REPLICATION_EVENT);
					simulator.addListener(this,
							SimulatorInterface.TIME_CHANGED_EVENT);
					this.treatmentField.setText("Treat(-/-)");
					this.replicationField.setText("Rep(-/-)");
					this.timeField.setText("Time:");
					if (simulator.getReplication() != null)
					{
						this.notify(new Event(
								SimulatorInterface.START_REPLICATION_EVENT,
								simulator, simulator.getReplication()));
					}
					this.validate();
					this.repaint();
				}
				return;
			}
		}
		if (event.getSource() instanceof SimulatorInterface)
		{
			if (event.getType().equals(
					SimulatorInterface.START_REPLICATION_EVENT))
			{
				this.units = this.application.getExperiment().getSimulator()
						.getReplication().getRunControl().getTreatment()
						.getTimeUnit().toString();
				int rep = ((SimulatorInterface) event.getSource())
						.getReplication().getNumber() + 1;
				int nrrep = ((SimulatorInterface) event.getSource())
						.getReplication().getRunControl().getReplications().length;
				int treat = ((SimulatorInterface) event.getSource())
						.getReplication().getRunControl().getTreatment()
						.getNumber() + 1;
				int nrtreat = ((SimulatorInterface) event.getSource())
						.getReplication().getRunControl().getTreatment()
						.getExperiment().getTreatments().length;
				this.replicationField.setText("Rep(" + rep + "/" + nrrep + ")");
				this.treatmentField.setText("Treat(" + treat + "/" + nrtreat
						+ ")");
				this.validate();
				this.repaint();
				return;
			}
			if (event.getType().equals(SimulatorInterface.TIME_CHANGED_EVENT))
			{
				this.simulatorTime = ((Double) event.getContent())
						.doubleValue();
			}
		}
	}

	/**
	 * The TimeUpdater
	 */
	private class TimeUpdater extends Thread
	{

		/** formatter */
		private NumberFormat formatter = NumberFormat.getInstance();

		/** the last length of the string */
		private int maxLength = 0;

		/** the source */
		private Statusbar source = null;

		/**
		 * constructs a new
		 * 
		 * @param source the statusbar
		 */
		public TimeUpdater(final Statusbar source)
		{
			super();
			this.source = source;
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run()
		{
			while (true)
			{
				try
				{
					String formattedTime = "Time:"
							+ this.formatter.format(this.source
									.getSimulatorTime()) + " "
							+ this.source.getUnits();
					this.source.getTimeField().setText(formattedTime);
					if (formattedTime.length() > this.maxLength)
					{
						this.maxLength = formattedTime.length();
						this.source.validate();
					}
					Thread.sleep(250);
				} catch (Throwable throwable)
				{
					throwable = null;
				}
			}
		}
	}
}