/*
 * @(#) DSOLFrame.java Oct 13, 2003
 * 
 * Copyright (c) 2002-2005 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands. All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.windows;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.gui.UIInitializer;
import nl.tudelft.simulation.dsol.gui.menu.HelpMenu;
import nl.tudelft.simulation.dsol.gui.panels.ControlPanel;
import nl.tudelft.simulation.dsol.gui.panels.ExperimentTree;
import nl.tudelft.simulation.dsol.gui.panels.Statusbar;
import nl.tudelft.simulation.dsol.gui.panels.TabbedPane;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.language.io.URLResource;
import nl.tudelft.simulation.language.reflection.ClassUtil;
import nl.tudelft.simulation.logger.Logger;

import org.jdom.Element;
import org.jdom.input.SAXBuilder;

/**
 * DSOL Frame <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision$ $Date$
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja">Peter Jacobs </a>
 */
public class DSOLFrame extends JFrame implements EventListenerInterface
{
	/** builder the xerces parser with validation turned on */
	private static SAXBuilder builder = new SAXBuilder(
			"org.apache.xerces.parsers.SAXParser", true);

	static
	{
		// turns on Schema Validation with Xerces
		builder.setFeature("http://xml.org/sax/features/validation", true);
		builder.setFeature("http://apache.org/xml/features/validation/schema",
				true);

		// Let's find the XSD file
		String xsd = URLResource.getResource("/navigation.xsd")
				.toExternalForm();
		builder
				.setProperty(
						"http://apache.org/xml/properties/schema/external-schemaLocation",
						"http://www.simulation.tudelft.nl/dsol " + xsd);
	}

	/** the preferredSize is the preferredSize of the panel */
	public static final Dimension PREFERRED_SIZE = new Dimension(400, 375);

	/** the application */
	protected DSOLApplicationInterface application = null;

	/** tabbedPanel */
	protected TabbedPane tabbedPanel = null;

	/** the experimentTree */
	protected JTree experimentTree = null;

	/** the statusbar */
	protected Statusbar statusbar = null;

	/** the controlPanel */
	protected ControlPanel controlPanel = null;

	/**
	 * constructs a new DSOLFrame
	 * 
	 * @param application the application
	 * @param navigation the URL address of the xml-based navigation structure.
	 */
	public DSOLFrame(final DSOLApplicationInterface application,
			final URL navigation)
	{
		super("DSOL main window");

		try
		{
			UIManager
					.setLookAndFeel("com.jgoodies.plaf.plastic.Plastic3DLookAndFeel");

			// The custom DSOL Color Look&Feel
			UIInitializer.initialize();
			Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
			this.setLocation((int) Math.round(screenSize.width * 0.5 - 0.5
					* PREFERRED_SIZE.width), (int) Math.round(screenSize.height
					* 0.33 - 0.5 * PREFERRED_SIZE.height));
			this.application = application;
			this.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
		} catch (Exception exception)
		{
			Logger.warning(this, "DSOLFrame", exception);
		}
		this.initialize(navigation);
		this.pack();
		this.setVisible(true);
	}

	/**
	 * initializes the DSOL Frame
	 * 
	 * @param navigation the navigation file
	 */
	private void initialize(final URL navigation)
	{
		JPanel contentPane = new JPanel();
		contentPane.setPreferredSize(PREFERRED_SIZE);
		contentPane.setLayout(new BorderLayout());
		this.setContentPane(contentPane);

		this.setJMenuBar(this.createMenuBar(navigation));

		this.controlPanel = new ControlPanel(this.application);

		this.tabbedPanel = new TabbedPane(SwingConstants.TOP);
		this.tabbedPanel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createEtchedBorder(), BorderFactory
						.createEmptyBorder()));
		contentPane.add(this.tabbedPanel, BorderLayout.CENTER);
		this.statusbar = new Statusbar(this.application);

		this.experimentTree = new JTree();
		this.experimentTree.setEditable(false);
		this.experimentTree.setFocusable(false);

		URL about = URLResource.getResource("/html/about.html");
		JEditorPane aboutPane = null;
		try
		{
			aboutPane = new JEditorPane(about);
		} catch (IOException exception)
		{
			Logger.warning(this, "initialize", exception);
		}
		aboutPane.setAlignmentY(Component.BOTTOM_ALIGNMENT);
		aboutPane.setEditable(false);
		this.tabbedPanel.add("About", aboutPane);
	}

	/**
	 * creates a menu bar
	 * 
	 * @param navigation the xml-based navigation
	 * 
	 * @return the menu bar
	 */
	private JMenuBar createMenuBar(final URL navigation)
	{
		Logger.info(this, "createMenuBar", "using "
				+ navigation.toExternalForm());
		try
		{
			JMenuBar menuBar = new JMenuBar();
			menuBar.setBorder(BorderFactory.createLineBorder(Color.GRAY));

			Element rootElement = builder.build(navigation).getRootElement();
			for (Iterator i = rootElement.getChildren("menu").iterator(); i
					.hasNext();)
			{
				Element menu = (Element) i.next();
				menuBar.add(this.parseMenu(menu));
			}
			menuBar.add(new HelpMenu(this.application));
			return menuBar;
		} catch (Exception exception)
		{
			Logger.warning(this, "initialze2", exception.getMessage());
			return null;
		}
	}

	/**
	 * parses an xml-based element into a JMenu
	 * 
	 * @param element the JDOM element
	 * @return the JMenu
	 * @throws Exception on IOException
	 */
	private JMenu parseMenu(final Element element) throws Exception
	{
		JMenu menu = new JMenu(element.getAttributeValue("name"));
		for (Iterator i = element.getChildren().iterator(); i.hasNext();)
		{
			Element child = (Element) i.next();
			if (child.getName().equals("menu"))
			{
				menu.add(this.parseMenu(child));
			} else if (child.getName().equals("separator"))
			{
				menu.add(new JSeparator());
			} else if (child.getName().equals("action"))
			{
				Class actionClass = Class.forName(child
						.getAttributeValue("className"));
				List arguments = new ArrayList();
				arguments.add(this.application);
				for (Iterator ii = child.getChildren("argument").iterator(); ii
						.hasNext();)
				{
					arguments.add(((Element) ii.next()).getValue());
				}
				Class[] argumentClasses = new Class[arguments.size()];
				argumentClasses[0] = DSOLApplicationInterface.class;
				for (int loop = 0; loop < argumentClasses.length - 1; loop++)
				{
					argumentClasses[loop + 1] = String.class;
				}
				Constructor actionConstructor = ClassUtil.resolveConstructor(
						actionClass, argumentClasses);
				Object action = actionConstructor.newInstance(arguments
						.toArray());
				if (action instanceof Action)
				{
					menu.add((Action) action);
				} else if (action instanceof JMenu)
				{
					menu.add((JMenu) action);
				}
			}
		}
		return menu;
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event)
	{
		if (event.getSource().equals(this.application)
				&& event.getType().equals(
						DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT))
		{
			this.tabbedPanel.remove("Description");
			this.tabbedPanel.remove("Experiment");
			this.tabbedPanel.remove("Control");
			if (event.getContent() != null)
			{
				Experiment experiment = (Experiment) event.getContent();
				try
				{
					String descriptionName = "/";
					if (experiment.getModel().getClass().getPackage() != null)
					{
						descriptionName = descriptionName
								+ experiment.getModel().getClass().getPackage()
										.getName().replace('.', '/');
					}
					descriptionName = descriptionName + "/package.html";
					URL description = experiment.getModel().getClass()
							.getResource(descriptionName);
					JEditorPane descriptionPane = null;
					descriptionPane = new JEditorPane(description);
					descriptionPane.setAlignmentY(Component.BOTTOM_ALIGNMENT);
					descriptionPane.setEditable(false);
					this.tabbedPanel.add("Description", 0, new JScrollPane(
							descriptionPane));
				} catch (IOException exception)
				{
					Logger
							.fine(this, "initialize",
									"experiment has no package.html describing the experiment");
				}
				this.setTitle("DSOL main window ("
						+ experiment.getProperty(Experiment.EXPERIMENT_NAME)
						+ ")");
				this.experimentTree.setModel(new ExperimentTree(experiment));
				this.tabbedPanel.add("Experiment", 1, new JScrollPane(
						this.experimentTree));
				this.tabbedPanel.add("Control", 2, this.controlPanel);
				this.getContentPane().add(this.statusbar, BorderLayout.SOUTH);
			} else
			{
				this.setTitle("DSOL main window");
				this.getContentPane().remove(this.statusbar);
			}
			this.validate();
			this.repaint();
		}
	}
}