/*
 * @(#) ContextFrame.java Oct 26, 2003
 * 
 * Copyright (c) 2002-2005 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands. All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.dsol.gui.windows;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Graphics;

import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.WindowConstants;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;

import nl.tudelft.simulation.dsol.eventlists.TableModelEventList;
import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.gui.DSOLApplicationInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.logger.Logger;

/**
 * The EventlistFrame <br>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty.
 * 
 * @version $Revision$ $Date$
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja">Peter Jacobs </a>
 */
public class EventlistFrame extends JFrame implements EventListenerInterface
{
	/** the application */
	private DSOLApplicationInterface application = null;

	/** the currently active simulator */
	private DEVSSimulatorInterface simulator = null;

	/** the table */
	private SynchronizedJTable table = new SynchronizedJTable(
			new DefaultTableModel(TableModelEventList.HEADER, 0));

	/**
	 * constructs a new ContextFrame
	 * 
	 * @param application the application to see
	 */
	public EventlistFrame(final DSOLApplicationInterface application)
	{
		super("Eventlist");
		this.application = application;
		this.initialize();
		try
		{
			this.application.addListener(this,
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT);
			this.notify(new Event(
					DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT,
					this.application, this.application.getExperiment()));
		} catch (Exception exception)
		{
			Logger.warning(this, "EventlistFrame", exception);
		}

		this.pack();
		this.setVisible(true);
		this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
	}

	/**
	 * @see java.awt.Window#dispose()
	 */
	public void dispose()
	{
		try
		{
			if (this.simulator != null
					&& this.simulator.getEventList() instanceof TableModelEventList)
			{
				boolean stopped = false;
				if (this.simulator.isRunning())
				{
					stopped = true;
					this.simulator.stop();
				}
				synchronized (this.simulator)
				{
					this.simulator.removeListener(this,
							DEVSSimulatorInterface.EVENTLIST_CHANGED_EVENT);
					this.undoSwapEventList();
					if (stopped)
					{
						this.simulator.start();
					}
				}
			}
		} catch (Exception exception)
		{
			Logger.warning(this, "dispose", exception);
		}
		super.dispose();
	}

	/**
	 * initializes the contextframe
	 */
	private void initialize()
	{
		JPanel panel = new JPanel(new BorderLayout());
		panel.setBorder(BorderFactory.createEtchedBorder());
		panel.add(new JScrollPane(this.table), BorderLayout.CENTER);
		panel.setPreferredSize(new Dimension(700, 300));
		this.setContentPane(panel);
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public void notify(final EventInterface event)
	{
		try
		{
			if (event.getSource().equals(this.application)
					&& event.getType().equals(
							DSOLApplicationInterface.EXPERIMENT_CHANGED_EVENT))
			{
				if (event.getContent() != null)
				{
					Experiment experiment = (Experiment) event.getContent();
					experiment.addListener(this,
							Experiment.SIMULATOR_CHANGED_EVENT);
					this.notify(new Event(Experiment.SIMULATOR_CHANGED_EVENT,
							experiment, experiment.getSimulator()));
				}
				return;
			}
			if (event.getSource() instanceof Experiment
					&& event.getType().equals(
							Experiment.SIMULATOR_CHANGED_EVENT))
			{
				if (event.getContent() != null
						&& event.getContent() instanceof DEVSSimulatorInterface)
				{
					this.simulator = (DEVSSimulatorInterface) event
							.getContent();
					this.simulator.addListener(this,
							DEVSSimulatorInterface.EVENTLIST_CHANGED_EVENT);
					this.notify(new Event(
							DEVSSimulatorInterface.EVENTLIST_CHANGED_EVENT,
							this.simulator, this.simulator.getEventList()));
				}
			}
			if (event.getSource().equals(this.simulator)
					&& event.getType().equals(
							DEVSSimulatorInterface.EVENTLIST_CHANGED_EVENT))
			{
				if (!(this.simulator.getEventList() instanceof TableModelEventList))
				{
					this.swapEventList();
				} else
				{
					this.table.setModel(((TableModelEventList) this.simulator
							.getEventList()).getTableModel());
				}
			}
			return;
		} catch (Exception exception)
		{
			Logger.warning(this, "dispose", exception);
		}
	}

	/**
	 * undoes the eventlistSwap
	 * 
	 * @throws Exception on failure
	 */
	private void undoSwapEventList() throws Exception
	{
		if (this.simulator != null)
		{
			boolean stopped = false;
			if (this.simulator.isRunning())
			{
				stopped = true;
				this.simulator.stop();
			}
			synchronized (this.simulator)
			{
				if (this.simulator.getEventList() instanceof TableModelEventList)
				{
					this.simulator
							.setEventList(((TableModelEventList) this.simulator
									.getEventList()).getOrigin());
				}
				this.table.setModel(new DefaultTableModel(
						TableModelEventList.HEADER, 0));
				if (stopped)
				{
					this.simulator.start();
				}
			}
		}
	}

	/**
	 * swaps the eventList
	 * 
	 * @throws Exception on exception
	 */
	private void swapEventList() throws Exception
	{
		if (this.simulator != null)
		{
			boolean stopped = false;
			if (this.simulator.isRunning())
			{
				stopped = true;
				this.simulator.stop();
			}
			synchronized (this.simulator)
			{
				TableModelEventList eventList = null;
				synchronized (this.simulator)
				{
					eventList = new TableModelEventList(this.simulator
							.getEventList());
					this.simulator.setEventList(eventList);
				}
				this.table.setModel(eventList.getTableModel());
				if (stopped)
				{
					this.simulator.start();
				}
			}
		}
	}

	/**
	 * Synchronizes the tableModel when updating
	 */
	private static class SynchronizedJTable extends JTable
	{
		/**
		 * constructs a new SynchronizedJTable
		 * 
		 * @param dm the tablemodel
		 */
		public SynchronizedJTable(final TableModel dm)
		{
			super(dm);
		}

		/**
		 * @see java.awt.Component#paint(java.awt.Graphics)
		 */
		public void paint(final Graphics g)
		{
			synchronized (this.getModel())
			{
				super.paint(g);
			}
		}

		/**
		 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
		 */
		protected void paintComponent(final Graphics g)
		{
			synchronized (this.getModel())
			{
				super.paintComponent(g);
			}
		}
	}
}