/*
 * @(#) MM1Queue.java Sep 21, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.dsol.tutorial.section41;

import java.rmi.RemoteException;

import nl.tudelft.simulation.dsol.ModelInterface;
import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.formalisms.Resource;
import nl.tudelft.simulation.dsol.formalisms.flow.Delay;
import nl.tudelft.simulation.dsol.formalisms.flow.Generator;
import nl.tudelft.simulation.dsol.formalisms.flow.Release;
import nl.tudelft.simulation.dsol.formalisms.flow.Seize;
import nl.tudelft.simulation.dsol.formalisms.flow.StationInterface;
import nl.tudelft.simulation.dsol.formalisms.flow.statistics.Utilization;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.dsol.statistics.Tally;
import nl.tudelft.simulation.dsol.statistics.charts.BoxAndWhiskerChart;
import nl.tudelft.simulation.jstats.distributions.DistConstant;
import nl.tudelft.simulation.jstats.distributions.DistContinuous;
import nl.tudelft.simulation.jstats.distributions.DistDiscreteConstant;
import nl.tudelft.simulation.jstats.distributions.DistExponential;
import nl.tudelft.simulation.jstats.streams.StreamInterface;

/**
 * The M/M/1 example as published in Simulation Modeling and Analysis by A.M.
 * Law & W.D. Kelton section 1.4 and 2.4.
 * <p>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>
 */
public class MM1Queue implements ModelInterface
{
	/**
	 * constructor for the MM1Queue
	 */
	public MM1Queue()
	{
		super();
	}

	/**
	 * @see nl.tudelft.simulation.dsol.ModelInterface#
	 *      constructModel(SimulatorInterface)
	 */
	public void constructModel(final SimulatorInterface simulator)
			throws SimRuntimeException, RemoteException
	{
		DEVSSimulatorInterface devsSimulator = (DEVSSimulatorInterface) simulator;

		StreamInterface defaultStream = devsSimulator.getReplication()
				.getStream("default");

		//The Generator
		Generator generator = new Generator(devsSimulator, Object.class, null);
		generator.setInterval(new DistExponential(defaultStream, 1.0));
		generator.setStartTime(new DistConstant(defaultStream, 0.0));
		generator.setBatchSize(new DistDiscreteConstant(defaultStream, 1));
		generator.setMaxNumber(1000);

		//The queue, the resource and the release
		Resource resource = new Resource(devsSimulator, 1.0);

		// created a resource
		StationInterface queue = new Seize(devsSimulator, resource);
		StationInterface release = new Release(devsSimulator, resource, 1.0);

		//The server
		DistContinuous serviceTime = new DistExponential(defaultStream, 0.5);
		StationInterface server = new Delay(devsSimulator, serviceTime);

		//The flow
		generator.setDestination(queue);
		queue.setDestination(server);
		server.setDestination(release);

		//Statistics
		Tally dN = new Tally("d(n)", devsSimulator, queue, Seize.DELAY_TIME);
		Tally qN = new Tally("q(n)", devsSimulator, queue,
				Seize.QUEUE_LENGTH_EVENT);
		Utilization uN = new Utilization("u(n)", devsSimulator, server);

		//Charts
		new BoxAndWhiskerChart(devsSimulator, "d(n) chart").add(dN);
		new BoxAndWhiskerChart(devsSimulator, "q(n) chart").add(qN);
		new BoxAndWhiskerChart(devsSimulator, "u(n) chart").add(uN);
	}
}