/*
 * @(#)ExperimentParser.java Aug 18, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.xml.dsol;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import nl.tudelft.simulation.dsol.ModelInterface;
import nl.tudelft.simulation.dsol.experiment.Experiment;
import nl.tudelft.simulation.dsol.experiment.Replication;
import nl.tudelft.simulation.dsol.experiment.RunControl;
import nl.tudelft.simulation.dsol.experiment.TimeUnitInterface;
import nl.tudelft.simulation.dsol.experiment.Treatment;
import nl.tudelft.simulation.jstats.streams.Java2Random;
import nl.tudelft.simulation.jstats.streams.StreamInterface;
import nl.tudelft.simulation.language.io.URLResource;
import nl.tudelft.simulation.logger.Logger;

import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * The ExperimentParser parses xml-based experiments into their java objects
 * <br>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version 2.0 21.09.2003 <br>
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>
 */
public class ExperimentParser
{

	/** builder the xerces parser with validation turned on */
	private static SAXBuilder builder = new SAXBuilder(
			"org.apache.xerces.parsers.SAXParser", true);
	static
	{
		//turns on Schema Validation with Xerces
		builder.setFeature("http://xml.org/sax/features/validation", true);
		builder.setFeature("http://apache.org/xml/features/validation/schema",
				true);
		//Let's find the XSD file
		String xsd = URLResource.getResource(
				"/nl/tudelft/simulation/xml/xsd/experiment.xsd")
				.toExternalForm();
		builder
				.setProperty(
						"http://apache.org/xml/properties/schema/external-schemaLocation",
						"http://www.simulation.tudelft.nl " + xsd);
	}

	/**
	 * constructs a new ExperimentParser This is a Utility Class.
	 */
	protected ExperimentParser()
	{
		//A utility class should not be instantiated
	}

	/**
	 * parses an experiment xml-file.
	 * 
	 * @param input the inputstream
	 * @return Experiment the experiment
	 * @throws IOException whenever parsing fails
	 */
	public static Experiment parseExperiment(final URL input)
			throws IOException
	{
		if (input == null)
		{
			throw new IOException("experiment URL=null");
		}
		try
		{
			ClassLoader loader = ExperimentParser.resolveClassLoader(input);
			Element rootElement = builder.build(input).getRootElement();
			Experiment experiment = new Experiment();
			if (rootElement.getChild("name") != null)
			{
				experiment.setProperty(Experiment.EXPERIMENT_NAME, rootElement
						.getChildText("name"));
			}
			if (rootElement.getChild("analyst") != null)
			{
				experiment.setProperty(Experiment.EXPERIMENT_ANALYST,
						rootElement.getChildText("analyst"));
			}
			Element modelElement = rootElement.getChild("model");

			if (modelElement.getChild("class-path") != null)
			{
				List jarFiles = modelElement.getChild("class-path")
						.getChildren("jar-file");
				URL[] urls = new URL[jarFiles.size()];
				int nr = 0;
				for (Iterator i = jarFiles.iterator(); i.hasNext();)
				{
					Element child = (Element) i.next();
					urls[nr] = URLResource.getResource(child.getValue());
					nr++;
				}
				loader = new URLClassLoader(urls, loader);
			}
			Thread.currentThread().setContextClassLoader(loader);

			experiment.setUrl(input);
			Class modelClass = Class.forName(modelElement
					.getChildText("model-class"), true, loader);

			ModelInterface model = (ModelInterface) modelClass.getConstructor(
					null).newInstance(null);
			experiment.setModel(model);
			List treatmentList = rootElement.getChildren("treatment");
			int number = 0;
			ArrayList treatmentArray = new ArrayList();
			for (Iterator i = treatmentList.iterator(); i.hasNext();)
			{
				Treatment treatment = ExperimentParser.parseTreatment(
						(Element) i.next(), experiment, number);
				number++;
				treatmentArray.add(treatment);
			}
			Treatment[] treatments = new Treatment[treatmentArray.size()];
			experiment.setTreatments((Treatment[]) treatmentArray
					.toArray(treatments));
			return experiment;
		} catch (Exception exception)
		{
			Logger
					.warning(ExperimentParser.class, "parseExperiment",
							exception);
			throw new IOException(exception.getMessage());
		}
	}

	//*********************** PRIVATE PARSING FUNCTIONS *******************//
	/**
	 * parses an experiment xml-file.
	 * 
	 * @param input the inputstream
	 * @return Classloader the classLoader
	 */
	private static ClassLoader resolveClassLoader(final URL input)
	{
		ClassLoader loader = Thread.currentThread().getContextClassLoader();
		if (input.getProtocol().equals("file"))
		{
			try
			{
				List urls = new ArrayList();
				String path = input.getPath().substring(0,
						input.getPath().lastIndexOf('/'))
						+ "/";
				//We add the path
				urls.add(new URL("file:" + path));
				//If the classpath ends with src, we also add the bin
				if (path.endsWith("src/"))
				{
					path = path.substring(0, path.length() - 4) + "bin/";
					urls.add(new URL("file:" + path));
				} else
				{
					//We assume there might be a src & bin dir
					path = path.substring(0, path.length()) + "bin/";
					urls.add(new URL("file:" + path));
				}
				URL[] urlArray = (URL[]) urls.toArray(new URL[urls.size()]);
				URLClassLoader urlClassLoader = new URLClassLoader(urlArray,
						loader);
				return urlClassLoader;
			} catch (MalformedURLException exception)
			{
				return loader;
			}
		}
		return loader;
	}

	/**
	 * parses the dateTime
	 * 
	 * @param value the string value in the yyyy-mm-ddThh:mm:ss format
	 * @return long the amount of milliseconds since 1970.
	 */
	private static long parseDateTime(final String value)
	{
		Calendar calendar = Calendar.getInstance();
		String concatDate = value.split("T")[0];
		String concatTime = value.split("T")[1];
		String[] date = concatDate.split("-");
		String[] time = concatTime.split(":");
		calendar.set(new Integer(date[0]).intValue(), new Integer(date[1])
				.intValue() - 1, new Integer(date[2]).intValue(), new Integer(
				time[0]).intValue(), new Integer(time[1]).intValue(),
				new Integer(time[2]).intValue());
		return calendar.getTimeInMillis();
	}

	/**
	 * parses a period
	 * 
	 * @param element the xml-element representing the period
	 * @param treatmentTimeUnit the timeUnit of the treatment
	 * @return double the value in units defined by the treatment
	 * @throws Exception whenever the period.
	 */
	private static double parsePeriod(final Element element,
			final TimeUnitInterface treatmentTimeUnit) throws Exception
	{
		TimeUnitInterface timeUnit = ExperimentParser.parseTimeUnit(element
				.getAttribute("unit").getValue());
		double value = -1;
		if (element.getText().equals("INF"))
		{
			value = Double.MAX_VALUE;
		} else
		{
			value = new Double(element.getText()).doubleValue();
		}
		if (value < 0)
		{
			throw new JDOMException("parsePeriod: value = " + value
					+ " <0. simulator cannot schedule in past");
		}
		return timeUnit.getValue() * value / treatmentTimeUnit.getValue();
	}

	/**
	 * parses a replication
	 * 
	 * @param element the JDOM element
	 * @param parent the RunControl
	 * @param number the number
	 * @return the replication
	 * @throws Exception on failure
	 */
	private static Replication parseReplication(final Element element,
			final RunControl parent, final int number) throws Exception
	{
		Replication replication = new Replication(parent, number);
		if (element.getAttribute("description") != null)
		{
			replication.setDescription(element.getAttribute("description")
					.getValue());
		}
		Map streams = new HashMap();
		List streamElements = element.getChildren("stream");
		for (Iterator i = streamElements.iterator(); i.hasNext();)
		{
			Element streamElement = (Element) i.next();
			long seed = new Long(streamElement.getAttributeValue("seed"))
					.longValue();
			StreamInterface stream = new Java2Random(seed);
			streams.put(streamElement.getAttributeValue("name"), stream);
		}
		replication.setStreams(streams);
		return replication;
	}

	/**
	 * parses proprties to treatments
	 * 
	 * @param element the element
	 * @return Properties
	 */
	private static Properties parseProperties(final Element element)
	{
		Properties result = new Properties();
		List children = element.getChildren("property");
		for (Iterator i = children.iterator(); i.hasNext();)
		{
			Element child = (Element) i.next();
			String key = child.getAttributeValue("key");
			String value = child.getAttributeValue("value");
			result.put(key, value);
		}
		return result;
	}

	/**
	 * parses the runcontrol
	 * 
	 * @param element the element
	 * @param parent the treatment
	 * @return RunControl result
	 * @throws Exception on failure
	 */
	private static RunControl parseRunControl(final Element element,
			final Treatment parent) throws Exception
	{
		RunControl runControl = new RunControl(parent);
		if (element.getChild("warmupPeriod") != null)
		{
			runControl.setWarmupPeriod(ExperimentParser.parsePeriod(element
					.getChild("warmupPeriod"), parent.getTimeUnit()));
		}
		if (element.getChild("runLength") != null)
		{
			runControl.setRunLength(ExperimentParser.parsePeriod(element
					.getChild("runLength"), parent.getTimeUnit()));
		}
		List replicationElements = element.getChildren("replication");
		ArrayList replicationArray = new ArrayList();
		int number = 0;
		for (Iterator i = replicationElements.iterator(); i.hasNext();)
		{
			replicationArray.add(ExperimentParser.parseReplication((Element) i
					.next(), runControl, number));
			number++;
		}
		Replication[] replications = new Replication[replicationArray.size()];
		runControl.setReplications((Replication[]) replicationArray
				.toArray(replications));
		return runControl;
	}

	/**
	 * parses a timeUnit
	 * 
	 * @param name the name
	 * @return TimeUnitInterface result
	 * @throws Exception on failure
	 */
	private static TimeUnitInterface parseTimeUnit(final String name)
			throws Exception
	{
		if (name.equals("DAY"))
		{
			return TimeUnitInterface.DAY;
		}
		if (name.equals("HOUR"))
		{
			return TimeUnitInterface.HOUR;
		}
		if (name.equals("MILLISECOND"))
		{
			return TimeUnitInterface.MILLISECOND;
		}
		if (name.equals("MINUTE"))
		{
			return TimeUnitInterface.MINUTE;
		}
		if (name.equals("SECOND"))
		{
			return TimeUnitInterface.SECOND;
		}
		if (name.equals("WEEK"))
		{
			return TimeUnitInterface.WEEK;
		}
		if (name.equals("UNIT"))
		{
			return TimeUnitInterface.UNIT;
		}
		throw new Exception("parseTimeUnit.. unknown argument: " + name);
	}

	/**
	 * parses the treatment
	 * 
	 * @param element the xml-element
	 * @param parent parent
	 * @param number the number
	 * @return Treatment
	 * @throws Exception on failure
	 */
	private static Treatment parseTreatment(final Element element,
			final Experiment parent, final int number) throws Exception
	{
		Treatment treatment = new Treatment(parent, number);
		treatment.setTimeUnit(ExperimentParser.parseTimeUnit(element
				.getChildText("timeUnit")));
		if (element.getChild("startTime") != null)
		{
			treatment.setStartTime(ExperimentParser.parseDateTime(element
					.getChildText("startTime")));
		}
		treatment.setRunControl(ExperimentParser.parseRunControl(element
				.getChild("runControl"), treatment));
		if (element.getChild("properties") != null)
		{
			treatment.setProperties(ExperimentParser.parseProperties(element
					.getChild("properties")));
		}
		return treatment;
	}
}