/*
 * @(#) ObjectJTable.java Apr 15, 2004
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.introspection.gui;

import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;

import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;

import nl.tudelft.simulation.introspection.mapping.CellPresentationConfiguration;
import nl.tudelft.simulation.introspection.mapping.DefaultConfiguration;
import nl.tudelft.simulation.introspection.sortable.SortDefinition;
import nl.tudelft.simulation.introspection.sortable.SortingTableHeader;

/**
 * * A customization of a standard JTable to allow the display of an
 * introspected object. The behaviour of the ObjectJTable depends on the
 * contained TableModel. {see ObjectTableModel}provides a view of the properties
 * and values of a single introspected object. {see
 * CollectionTableModel}provides a view on a collection of instances: usually
 * the value of a composite property.
 * <p>
 * A configuration mechanism is implemented to load the editors and renders to
 * be used by this JTable. See {see
 * #setConfig(nl.tudelft.simulation.introspection.mapping.CellPresentationConfiguration)}
 * for details.
 * <p>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl">www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a
 *         href="http://web.eur.nl/fbk/dep/dep1/Introduction/Staff/People/Lang">Niels
 *         Lang </a><a
 *         href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>
 * @version 1.1 Apr 15, 2004
 * @since 1.4
 */
public class ObjectJTable extends JTable implements ObjectJTableInterface
{
	/** the updateTimer */
	private static UpdateTimer updateTimer = new UpdateTimer(100L);

	/** hasShown? */
	protected boolean hasShown = false;

	/** the introspectionTableModel */
	private IntrospectingTableModelInterface introspectionTableModel;

	/**
	 * constructs a new ObjectJTable
	 * 
	 * @param dm the defaultTableModel
	 */
	public ObjectJTable(final IntrospectingTableModelInterface dm)
	{
		this(dm, DefaultConfiguration.getDefaultConfiguration());
	}

	/**
	 * constructs a new ObjectJTable
	 * 
	 * @param dm the defaultTableModel
	 * @param config the CellPresentationConfiguration
	 */
	public ObjectJTable(final IntrospectingTableModelInterface dm,
			final CellPresentationConfiguration config)
	{
		super(new SortingObjectTableModel(dm));
		init(config, dm);
	}

	/**
	 * Constructor for ObjectJTable.
	 * 
	 * @param dm the defaultTableModel
	 * @param cm the tableColumnModel
	 */
	public ObjectJTable(final IntrospectingTableModelInterface dm,
			final TableColumnModel cm)
	{
		super(new SortingObjectTableModel(dm), cm);
		init(DefaultConfiguration.getDefaultConfiguration(), dm);
	}

	/**
	 * Constructor for ObjectJTable.
	 * 
	 * @param dm the defaultTableModel
	 * @param cm the tableColumnModel
	 * @param sm the listSelectionModel
	 */
	public ObjectJTable(final IntrospectingTableModelInterface dm,
			final TableColumnModel cm, final ListSelectionModel sm)
	{
		super(new SortingObjectTableModel(dm), cm, sm);
		init(DefaultConfiguration.getDefaultConfiguration(), dm);
	}

	/**
	 * initializes the objectJTable
	 * 
	 * @param config the configuration
	 * @param model the model
	 */
	private void init(final CellPresentationConfiguration config,
			final IntrospectingTableModelInterface model)
	{
		this.introspectionTableModel = model;
		setConfig(config);
		setPreferredScrollableViewportSize(this.getPreferredSize());
		JTableHeader header = new SortingTableHeader(
				new SortDefinition[]{new SortDefinition(0, true)});
		this.setTableHeader(header);
		header.setColumnModel(this.getColumnModel());
		ObjectJTable.updateTimer.add(this);
	}

	/**
	 * the ParentListener
	 */
	private class ParentListener implements HierarchyListener
	{
		/**
		 * @see java.awt.event.HierarchyListener#hierarchyChanged(HierarchyEvent)
		 */
		public void hierarchyChanged(final HierarchyEvent e)
		{
			if (e.getChangeFlags() == HierarchyEvent.DISPLAYABILITY_CHANGED)
			{
				if (!ObjectJTable.this.hasShown && isDisplayable())
				{
					ObjectJTable.this.hasShown = true;
					return;
				}
				if (ObjectJTable.this.hasShown && !isDisplayable())
				{
					ObjectJTable.this.getModel().removeTableModelListener(
							ObjectJTable.this);
				}
			}
		}
	}

	/**
	 * sets the config
	 * 
	 * @param config the config
	 */
	private void setConfig(final CellPresentationConfiguration config)
	{
		addHierarchyListener(new ParentListener());
		Class[][] renderers = config.getRenderers();
		Class[][] editors = config.getEditors();
		try
		{
			for (int i = 0; i < renderers.length; i++)
			{
				this.setDefaultRenderer(renderers[i][0],
						(TableCellRenderer) renderers[i][1].newInstance());
			}
			for (int i = 0; i < editors.length; i++)
			{
				this.setDefaultEditor(editors[i][0],
						(TableCellEditor) editors[i][1].newInstance());
			}
		} catch (Exception e)
		{
			throw new IllegalArgumentException("Configuration " + config
					+ "failed, " + "probably invalid classes.");
		}
		this.getColumn(getColumnName(0)).setPreferredWidth(70);
		this.getColumn(getColumnName(1)).setMaxWidth(25);
		this.getColumn(getColumnName(2)).setPreferredWidth(450);
		this.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
	}

	/**
	 * @see nl.tudelft.simulation.introspection.gui.ObjectJTableInterface
	 *      #getIntrospectingTableModel()
	 */
	public IntrospectingTableModelInterface getIntrospectingTableModel()
	{
		return this.introspectionTableModel;
	}
}