/*
 * @(#) XYSeries.java Sep 26, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.jstats.charts.xy;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventListenerInterface;
import nl.tudelft.simulation.event.EventType;
import nl.tudelft.simulation.event.TimedEvent;
import nl.tudelft.simulation.logger.Logger;

import org.jfree.data.AbstractDataset;

/**
 * The xySerie specifies an xySerie for XY Plots in DSOL.
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.8 2004-03-18
 * @since 1.0
 */
public class XYSeries extends AbstractDataset implements EventListenerInterface
{
	/** LOWER_RANGE_EVENT is fired on a range change */
	public static final EventType LOWER_RANGE_EVENT = new EventType(
			"LOWER_RANGE_EVENT");

	/** UPPER_RANGE_EVENT is fired on a range change */
	public static final EventType UPPER_RANGE_EVENT = new EventType(
			"UPPER_RANGE_EVENT");

	/** name refers to the name of the serie */
	private String name = null;

	/** the entries of the serie */
	private List entries = new ArrayList();

	/** the axisType (default, logarithmic) */
	private short axisType = XYChart.XLINEAR_YLINEAR;

	/**
	 * constructs a new XYSeries.
	 * 
	 * @param name the name of the series.
	 * @param axisType whether this serie is logarithmic (x=0 & y=0 are
	 *        neglected)
	 */
	public XYSeries(final String name, final short axisType)
	{
		super();
		this.axisType = axisType;
		this.name = name;
		this.fireDatasetChanged();
	}

	/**
	 * @see nl.tudelft.simulation.event.EventListenerInterface
	 *      #notify(nl.tudelft.simulation.event.EventInterface)
	 */
	public synchronized void notify(final EventInterface event)
	{
		TimedEvent timedEvent = (TimedEvent) event;

		//We have chosen to simply neglect <=0.0 values on logarithmic axis
		if (this.axisType == XYChart.XLOGARITHMIC_YLINEAR
				|| this.axisType == XYChart.XLOGARITHMIC_YLOGARITHMIC)
		{
			if (timedEvent.getTimeStamp() <= 0.0)
			{
				Logger.warning(this, "notify", "refusing xvalue of " + event
						+ " on logrithmic chart");
				return;
			}
		}
		if (this.axisType == XYChart.XLINEAR_YLOGARITHMIC
				|| this.axisType == XYChart.XLOGARITHMIC_YLOGARITHMIC)
		{
			if (((Number) timedEvent.getContent()).doubleValue() <= 0.0)
			{
				Logger.warning(this, "notify", "refusing yValue of " + event
						+ " on logrithmic chart");
				return;
			}
		}
		this.entries.add(new Tuple(timedEvent.getTimeStamp(),
				(Number) timedEvent.getContent()));
		this.fireDatasetChanged();
	}

	/**
	 * returns the number of items in this seris
	 * 
	 * @return int the number
	 */
	public int getItemCount()
	{
		return this.entries.size();
	}

	/**
	 * returns the X value
	 * 
	 * @param item the item
	 * @return Number the xValue
	 */
	public Number getXValue(final int item)
	{
		return ((Tuple) this.entries.get(item)).getXValue();
	}

	/**
	 * returns the yValue
	 * 
	 * @param item the item
	 * @return Number
	 */
	public Number getYValue(final int item)
	{
		return ((Tuple) this.entries.get(item)).getYValue();
	}

	/**
	 * returns the name of this serie
	 * 
	 * @return String name
	 */
	public String getSeriesName()
	{
		return this.name;
	}

	/**
	 * holds the x-y value pairs as tuple
	 */
	private class Tuple implements Serializable
	{
		/** x-value is the xValue */
		private double xValue = Double.NaN;

		/** yValue is the yValue */
		private double yValue = Double.NaN;

		/**
		 * constructs a new tuple constructs a new Tuple
		 * 
		 * @param xValue the x-value.
		 * @param yValueNumber the y-value.
		 */
		public Tuple(final double xValue, final Number yValueNumber)
		{
			this.xValue = xValue;
			this.yValue = yValueNumber.doubleValue();
		}

		/**
		 * returns the xValue.
		 * 
		 * @return Number the xValue
		 */
		public Number getXValue()
		{
			return new Double(this.xValue);
		}

		/**
		 * returns the yValue.
		 * 
		 * @return Number the yValue
		 */
		public Number getYValue()
		{
			return new Double(this.yValue);
		}
	}
}