/*
 * @(#)DistGeometric.java Apr 3, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.jstats.distributions;

import nl.tudelft.simulation.jstats.streams.StreamInterface;

/**
 * The Geometric distribution. For more information on this distribution see <a
 * href="http://mathworld.wolfram.com/GeometricDistribution.html">
 * http://mathworld.wolfram.com/GeometricDistribution.html </a>
 * <p>
 * (c) copyright 2002-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">
 *         Alexander Verbraeck </a> <br>
 *         <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm"> Peter
 *         Jacobs </a>
 * @version 1.9 2004-03-22
 * @since 1.2
 */
public class DistGeometric extends DistDiscrete
{
	/** p is the p-value of the geometric distribution */
	private double p;

	/** lnp is a helper variable to avoid repetitive calculation */
	private double lnp;

	/**
	 * constructs a new geometric distribution
	 * 
	 * @param stream the numberstream
	 * @param p is the p-value
	 */
	public DistGeometric(final StreamInterface stream, final double p)
	{
		super(stream);
		if ((p > 0.0) && (p < 1.0))
		{
			this.p = p;
		} else
		{
			throw new IllegalArgumentException("Error Geometric - p<=0 or p>=1");
		}
		this.lnp = Math.log(1.0 - this.p);
	}

	/**
	 * @see DistDiscrete#draw()
	 */
	public long draw()
	{
		double u = this.stream.nextDouble();
		return (long) (Math.floor(Math.log(u) / this.lnp));
	}

	/**
	 * @see nl.tudelft.simulation.jstats.distributions.DistDiscrete
	 *      #probability(int)
	 */
	public double probability(final int observation)
	{
		if (observation >= 0)
		{
			return this.p * Math.pow(1 - this.p, observation);
		}
		return 0.0;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return "Geometric(" + this.p + ")";
	}
}