/*
 * @(#)DistUniform.java Apr 3, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.jstats.distributions;

import nl.tudelft.simulation.jstats.streams.StreamInterface;

/**
 * The Uniform distribution. For more information on this distribution see <a
 * href="http://mathworld.wolfram.com/UniformDistribution.html">
 * http://mathworld.wolfram.com/UniformDistribution.html </a>
 * <p>
 * (c) copyright 2002-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">
 *         Alexander Verbraeck </a> <br>
 *         <a href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm"> Peter
 *         Jacobs </a>
 * @version 1.9 2004-03-22
 * @since 1.2
 */
public class DistUniform extends DistContinuous
{
	/** a is the minimum */
	private double a;

	/** b is the maximum */
	private double b;

	/**
	 * constructs a new uniform distribution. a and b are real numbers with a
	 * less than b. a is a location parameter, b-a is a scale parameter.
	 * 
	 * @param stream the numberstream
	 * @param a the minimum value
	 * @param b the maximum value
	 */
	public DistUniform(final StreamInterface stream, final double a,
			final double b)
	{
		super(stream);
		this.a = a;
		if (b > a)
		{
			this.b = b;
		} else
		{
			throw new IllegalArgumentException("Error Uniform - a >= b");
		}
	}

	/**
	 * @see DistContinuous#draw()
	 */
	public double draw()
	{
		return this.a + (this.b - this.a) * this.stream.nextDouble();
	}

	/**
	 * @see nl.tudelft.simulation.jstats.distributions.DistContinuous
	 *      #probDensity(double)
	 */
	public double probDensity(final double observation)
	{
		if (observation >= this.a && observation <= this.b)
		{
			return 1.0 / (this.b - this.a);
		}
		return 0.0;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return "Uniform(" + this.a + "," + this.b + ")";
	}
}