/*
 * @(#) AbstractFilter.java Oct 26, 2004
 * 
 * Copyright (c) 2004 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.language.filters;

/**
 * The abstract filter forms the abstract class for all filters. The filter
 * method should be implemented by all subclasses. This filter method should
 * have the same semantics as the accept(inverted=false) method.
 * <p>
 * (c) copyright 2004 <a href="http://www.simulation.tudelft.nl/dsol/">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl/dsol/">
 * www.simulation.tudelft.nl/dsol </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a
 *         href="http://web.eur.nl/fbk/dep/dep1/Introduction/Staff/People/Lang">Niels
 *         Lang </a><a
 *         href="http://www.tbm.tudelft.nl/webstaf/peterja/index.htm">Peter
 *         Jacobs </a>
 * @version 1.0 Oct 26, 2004
 * @since 1.2
 */
public abstract class AbstractFilter implements Filterinterface
{
	/** is this filter inverted */
	protected boolean inverted = false;

	/**
	 * constructs a new AbstractFilter
	 */
	public AbstractFilter()
	{
		super();
	}

	/**
	 * @see nl.tudelft.simulation.language.filters.Filterinterface#isInverted()
	 */
	public boolean isInverted()
	{
		return this.inverted;
	}

	/**
	 * @see nl.tudelft.simulation.language.filters.Filterinterface
	 *      #setInverted(boolean)
	 */
	public void setInverted(final boolean inverted)
	{
		this.inverted = inverted;
	}

	/**
	 * @see nl.tudelft.simulation.language.filters.Filterinterface#accept(java.lang.Object)
	 */
	public boolean accept(final Object entry)
	{
		boolean value = this.filter(entry);
		if (!this.inverted)
		{
			return value;
		}
		return !value;
	}

	/**
	 * filters the entry. This method should be implemented by every filter
	 * based on its semantic meaning.
	 * 
	 * @param entry the entry to filter.
	 * 
	 * @return whether to accept the value.
	 */
	protected abstract boolean filter(Object entry);

	/**
	 * returns the filter criterium
	 * 
	 * @return the criterium
	 */
	public abstract String getCriterium();

	/**
	 * adds filter to this filter and returns the composed filter
	 * 
	 * @param filter the filter to add
	 * @return the composed filter
	 */
	public Filterinterface add(final Filterinterface filter)
	{
		return new CompositeFilter(this, filter, CompositeFilter.ADD);
	}

	/**
	 * creates a new composite filter which is one or two
	 * 
	 * @param filter the filter to add
	 * @return the composed filter
	 */
	public Filterinterface or(final Filterinterface filter)
	{
		return new CompositeFilter(this, filter, CompositeFilter.OR);
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return "Filter[criterium=" + this.getCriterium() + ";inverted="
				+ this.inverted + "]";
	}

}