/*
 * @(#) FieldSignature.java Jan 12, 2004 Copyright (c) 2002-2005 Delft
 * University of Technology Jaffalaan 5, 2628 BX Delft, the Netherlands. All
 * rights reserved. This software is proprietary information of Delft University
 * of Technology The code is published under the Lesser General Public License
 */
package nl.tudelft.simulation.language.reflection;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.HashMap;
import java.util.Map;

import nl.tudelft.simulation.language.primitives.Primitive;

/**
 * A field descriptor represents the type of a class, instance, or local
 * variable. It is a series of characters generated by the grammar described at
 * <a href =
 * "http://java.sun.com/docs/books/vmspec/2nd-edition/html/ClassFile.doc.html#1169">
 * The Java Virtual Machine Specification </a>.
 * <p>
 * (c) copyright 2002-2005 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands.
 * <p>
 * See for project information <a
 * href="http://www.simulation.tudelft.nl/dsol/language">www.simulation.tudelft.nl/language
 * </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/lesser.html">Lesser
 * General Public License (LGPL) </a>, no warranty
 * 
 * @author <a href="http://www.peter-jacobs.com/index.htm">Peter Jacobs </a>, <a
 *         href="mailto:nlang@fbk.eur.nl">Niels Lang </a><a
 *         href="mailto:a.verbraeck@tbm.tudelft.nl">Alexander
 *         Verbraeck </a>
 * @version $Revision: 1.8 $ $Date: 2005/08/04 12:08:54 $
 * @since 1.5
 */
public class FieldSignature implements Serializable
{
    /** the CAHCHE */
    private static final Map<String, Class> CACHE = new HashMap<String, Class>();

    /** the value of the field descriptor */
    private String value;

    /**
     * constructs a new FieldSignature
     * 
     * @param value the value of the descriptor
     */
    public FieldSignature(final String value)
    {
        super();
        this.value = value;
    }

    /**
     * constructs a new FieldSignature
     * 
     * @param clazz The class
     */
    public FieldSignature(final Class clazz)
    {
        this(FieldSignature.toDescriptor(clazz));
    }

    /**
     * @return Returns the value of the field descriptor
     */
    public String getStringValue()
    {
        return this.value;
    }

    /**
     * @return Returns the value of the field descriptor
     * @throws ClassNotFoundException if the class cannot be found.
     */
    public Class getClassValue() throws ClassNotFoundException
    {
        return FieldSignature.toClass(this.value);
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
	public String toString()
    {
        return this.value;
    }

    /**
     * converts an array of fields to its descriptor
     * 
     * @param classes the classes to represent
     * @return String the descriptor String
     */
    public static final String toDescriptor(final Class[] classes)
    {
        String result = "";
        for (int i = 0; i < classes.length; i++)
        {
            result = result + FieldSignature.toDescriptor(classes[i]);
        }
        return result;
    }

    /**
     * converts a field to its descriptor
     * 
     * @param clazz the clazz to represent
     * @return String the descriptor String
     */
    public static final String toDescriptor(final Class clazz)
    {
        if (clazz.getName().startsWith("["))
        {
            return clazz.getName().replace('.', '/');
        }
        if (clazz.isPrimitive())
        {
            if (clazz.equals(int.class))
            {
                return "I";
            }
            if (clazz.equals(double.class))
            {
                return "D";
            }
            if (clazz.equals(boolean.class))
            {
                return "Z";
            }
            if (clazz.equals(char.class))
            {
                return "C";
            }
            if (clazz.equals(byte.class))
            {
                return "B";
            }
            if (clazz.equals(float.class))
            {
                return "F";
            }
            if (clazz.equals(long.class))
            {
                return "J";
            }
            if (clazz.equals(short.class))
            {
                return "S";
            }
            return "V";
        }
        return "L" + clazz.getName().replace('.', '/') + ";";
    }

    /**
     * converts a fieldDescriptor to its class representation
     * 
     * @param descriptor the descriptor
     * @return Class the class
     * @throws ClassNotFoundException on failure
     */
    public static final Class toClass(final String descriptor)
            throws ClassNotFoundException
    {
        if (FieldSignature.CACHE.containsKey(descriptor))
        {
            return FieldSignature.CACHE.get(descriptor);
        }
        String className = descriptor;
        Class result = null;
        int array = 0;
        while (className.charAt(array) == '[')
        {
            array++;
        }
        className = className.substring(array);
        if (className.startsWith("L"))
        {
            className = className.replaceAll("/", ".");
            className = className.substring(1, className.length() - 1);
            try
            {
                result = Class.forName(className);
            } catch (Exception exception)
            {
                result = Class.forName(className, true, Thread.currentThread()
                        .getContextClassLoader());
            }
        } else
        {
            result = Primitive.forName(className);
        }
        if (result == null && !descriptor.startsWith("["))
        {
            // For some reason not all classes start with L and end with ;
            return FieldSignature.toClass("L" + descriptor + ";");
        }
        if (array == 0)
        {
            FieldSignature.CACHE.put(descriptor, result);
            return result;
        }
        try
        {
            int[] dimensions = new int[array];
            result = Array.newInstance(result, dimensions).getClass();
        } catch (Exception exception)
        {
            throw new ClassNotFoundException(result + " class not found");
        }
        FieldSignature.CACHE.put(descriptor, result);
        return result;
    }
}