/*
 * @(#) ContextUtil.java Oct 26, 2003
 * 
 * Copyright (c) 2003 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */
package nl.tudelft.simulation.naming.context;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;

import nl.tudelft.simulation.logger.Logger;

/**
 * ContextUtility class
 * <p>
 * (c) copyright 2003 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @author <a href="http://www.simulation.tudelft.nl/people/jacobs.html">Peter
 *         Jacobs </a>
 * @version 1.2 2004-03-24
 * @since 1.0
 */
public class ContextUtil
{
	/**
	 * constructs a new ContextUtil
	 */
	protected ContextUtil()
	{
		super();
	}

	/**
	 * resolves the name of an object under which it is accessible in the
	 * context
	 * 
	 * @param object the object
	 * @return String
	 * @throws NamingException whenever the object cannot be found
	 */
	public static String resolveKey(final Object object) throws NamingException
	{
		String result = ContextUtil
				.resolveKey(object, new InitialContext(), "");
		if (result == null)
		{
			throw new NamingException("could not resolve " + object.toString());
		}
		return result;
	}

	/**
	 * resolves the key under which an object is stored in the context.
	 * 
	 * @param object the object which key to resolve.
	 * @param context the context.
	 * @param name the name of the parent.
	 * @return the key
	 * @throws NamingException on lookup failure
	 */
	private static String resolveKey(final Object object,
			final Context context, final String name) throws NamingException
	{
		NamingEnumeration list = context.listBindings(name);
		while (list.hasMore())
		{
			Binding binding = (Binding) list.next();
			if (binding.getObject() instanceof Context)
			{
				String result = ContextUtil.resolveKey(object,
						(Context) binding.getObject(), binding.getName());
				if (result != null)
				{
					return result;
				}
			} else if (binding.getObject().equals(object))
			{
				String key = context.getNameInNamespace() + "/"
						+ binding.getName();
				return key;
			}
		}
		return null;
	}

	/**
	 * unbinds an object from the context
	 * 
	 * @param object the object to be removed.
	 */
	public static void unbindFromContext(Object object)
	{
		try
		{
			InitialContext context = new InitialContext();
			String key = ContextUtil.resolveKey(object, context, "/");
			context.unbind(key);
		} catch (NamingException namingException)
		{
			Logger.warning(ContextUtil.class, "unbindFromContext",
					namingException);
		}
	}
}