/*
 * @(#)SingleTrackBlockTrafficLight.java May 31, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.controlpoint.blocks;

import java.rmi.RemoteException;
import java.util.List;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.dsol.animation.LocatableInterface;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.animation.BlockTrafficLightAnimation;
import nl.tudelft.simulation.traffic.controlpoint.real.AbstractVisibleControlPoint;
import nl.tudelft.simulation.traffic.controlpoint.real.Changeable;
import nl.tudelft.simulation.traffic.controlpoint.real.StopSignInterface;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version May 31, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class SingleTrackBlockTrafficLight extends AbstractVisibleControlPoint
        implements
            LocatableInterface,
            StopSignInterface
{
    /** GREEN state */
    public static final String STATE_GREEN = "GREEN";

    /** YELLOW state */
    public static final String STATE_YELLOW = "YELLOW";

    /** RED state */
    public static final String STATE_RED = "RED";

    /** BLACK state */
    public static final String STATE_BLACK = "BLACK";

    /** GREEN sensor */
    public static final int SENSOR_GREEN = 0;

    /** YELLOW sensor */
    public static final int SENSOR_YELLOW = 1;

    /** RED sensor */
    public static final int SENSOR_RED = 2;

    /** sensor names */
    public static final String[] sensorNames = new String[]{"GREEN", "YELLOW",
            "RED"};

    /** the distance to the sensor AFTER the traffic light for RED */
    private double distanceRed;

    /** the distance to the sensor AFTER the traffic light for YELLOW */
    private double distanceYellow;

    /** the distance to the sensor AFTER the traffic light for GREEN */
    private double distanceGreen;

    /** the STATE of the current block signal */
    private String currentState = STATE_GREEN;

    /** the name */
    private String name;

    /** dx for animation (bounds) */
    private double dx;

    /** dy for animation (bounds) */
    private double dy;

    /**
     * @param name
     * @param progression
     * @param visibleDistance
     * @param track
     * @param distanceRed
     * @param distanceYellow
     * @param distanceGreen
     * @param simulator
     * @param dx
     * @param dy
     * @throws Exception
     */
    public SingleTrackBlockTrafficLight(final String name,
            final double progression, final double visibleDistance,
            final TrackInterface track, final double distanceRed,
            final double distanceYellow, final double distanceGreen,
            final DEVSSimulatorInterface simulator, final double dx,
            final double dy) throws Exception
    {
        super(track, progression, visibleDistance, simulator);
        this.name = name;
        System.out.println(this + ", placed on " + track + " progression "
                + progression + " (length=" + track.getLength() + ")");
        this.distanceRed = distanceRed;
        this.distanceYellow = distanceYellow;
        this.distanceGreen = distanceGreen;
        addSensors(super.track, progression + this.distanceRed, SENSOR_RED,
                simulator);
        addSensors(super.track, progression + this.distanceYellow,
                SENSOR_YELLOW, simulator);
        addSensors(super.track, progression + this.distanceGreen, SENSOR_GREEN,
                simulator);
        this.dx = dx;
        this.dy = dy;
        if (simulator instanceof AnimatorInterface)
        {
            new BlockTrafficLightAnimation(this, simulator, dx, dy);
        }
    }

    /**
     * @param track
     * @param place
     * @param sensor
     * @param simulator
     * @throws Exception
     */
    private void addSensors(final TrackInterface track, final double place,
            final int sensor, final SimulatorInterface simulator)
            throws Exception
    {
        List tpList = track.calculateTrackProgressionListAll(place);
        for (int i = 0; i < tpList.size(); i++)
        {
            TrackProgression tp = (TrackProgression) tpList.get(i);
            new SingleTrackControlPoint(tp.getTrack(), tp.getProgression(),
                    this, sensor, simulator);
            System.out.println(this + ", sensor "
                    + SingleTrackBlockTrafficLight.sensorNames[sensor]
                    + " added on track " + tp.getTrack() + ", progression "
                    + tp.getProgression());
        }
    }

    /**
     * The stateful handling of a sensor trigger. Works as follows:
     * <P>
     * 
     * <pre>
     * 
     *       -------------------------------
     *       | OLD___ | SENSOR___ | NEW___ |
     *       | STATE_ | R | Y | G | STATE_ |
     *       -------------------------------
     *       | BLACK_ | X | X | X | BLACK_ | (error)
     *       -------------------------------
     *       | RED___ | X | . | . | BLACK_ | (error)
     *       | RED___ | . | X | . | YELLOW |
     *       | RED___ | . | . | X | RED___ |
     *       -------------------------------
     *       | YELLOW | X | . | . | RED___ |
     *       | YELLOW | . | X | . | BLACK_ | (error)
     *       | YELLOW | . | . | X | GREEN_ |
     *       -------------------------------
     *       | GREEN_ | X | . | . | RED___ |
     *       | GREEN_ | . | X | . | BLACK_ | (error)
     *       | GREEN_ | . | . | X | BLACK_ | (error)
     *       -------------------------------
     *  
     * </pre>
     * 
     * @param sensor
     * @param vehicle
     */
    public void triggerSensor(final int sensor,
            final VehiclePhysicalInterface vehicle)
    {
        Logger.fine(this, "triggerSensor", this + ", Vehicle "
                + vehicle.toString() + " triggered sensor "
                + SingleTrackBlockTrafficLight.sensorNames[sensor]
                + ", old state was " + this.currentState);
        System.out.println(this + ", Vehicle " + vehicle.toString()
                + " triggered sensor "
                + SingleTrackBlockTrafficLight.sensorNames[sensor]
                + ", old state was " + this.currentState);
        if (this.currentState == STATE_BLACK)
        {
            this.currentState = STATE_BLACK;
        } else if (this.currentState == STATE_RED)
        {
            switch (sensor)
            {
                case SENSOR_RED :
                    this.currentState = STATE_BLACK;
                    Logger.severe(this, "triggerSensor", "RED - RED");
                    break;
                case SENSOR_YELLOW :
                    this.currentState = STATE_YELLOW;
                    break;
                case SENSOR_GREEN :
                    this.currentState = STATE_RED;
                    break;
                default :
                    Logger.severe(this, "triggerSensor",
                            "reported state not RED, YELLOW or GREEN");
                    break;
            }
        } else if (this.currentState == STATE_YELLOW)
        {
            switch (sensor)
            {
                case SENSOR_RED :
                    this.currentState = STATE_RED;
                    break;
                case SENSOR_YELLOW :
                    this.currentState = STATE_BLACK;
                    Logger.severe(this, "triggerSensor", "YELLOW - YELLOW");
                    break;
                case SENSOR_GREEN :
                    this.currentState = STATE_GREEN;
                    break;
                default :
                    Logger.severe(this, "triggerSensor",
                            "reported state not RED, YELLOW or GREEN");
                    break;
            }
        } else if (this.currentState == STATE_GREEN)
        {
            switch (sensor)
            {
                case SENSOR_RED :
                    this.currentState = STATE_RED;
                    break;
                case SENSOR_YELLOW :
                    this.currentState = STATE_BLACK;
                    Logger.severe(this, "triggerSensor", "GREEN - YELLOW");
                    break;
                case SENSOR_GREEN :
                    this.currentState = STATE_BLACK;
                    Logger.severe(this, "triggerSensor", "GREEN - GREEN");
                    break;
                default :
                    Logger.severe(this, "triggerSensor",
                            "reported state not RED, YELLOW or GREEN");
                    break;
            }
        } else
        {
            Logger
                    .severe(this, "triggerSensor",
                            "SingleTrackBlockTrafficLight not in state RED, YELLOW or GREEN");
        }
        Logger.fine(this, "triggerSensor", this + ", Vehicle "
                + vehicle.toString() + " new state is " + this.currentState);
        System.out.println(this + ", Vehicle " + vehicle.toString()
                + " new state is " + this.currentState);
        Event event = new Event(Changeable.CHANGE_STATUS_EVENT, this, null);
        this.fireEvent(event);
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#pass(nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface)
     */
    public void pass(final VehiclePhysicalInterface vehicle)
    {
        // cancel the listener of the vehicle
        this.removeListener(vehicle, Changeable.CHANGE_STATUS_EVENT);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        DirectedPoint p = super.getTrack().getLocationOfProgression(
                super.getProgression());
        return new DirectedPoint(p.x, p.y, p.z);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds() throws RemoteException
    {
        return new BoundingBox(new Point3d(this.dx - 2.0, this.dy, 0),
                new Point3d(this.dx + 2.0, this.dy + 12.0, 0));
    }

    /**
     * @return Returns the currentState.
     */
    public String getCurrentState()
    {
        return this.currentState;
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.real.Changeable#getStatus()
     */
    public String getStatus()
    {
        if ((this.currentState == STATE_RED)
                || (this.currentState == STATE_BLACK))
            return StopSignInterface.STOP;
        else
            return StopSignInterface.CONTINUE;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return "SingleBlockTrafficLight " + this.name;
    }
}