/*
 * @(#)SingleTrackBlockTrafficLight.java May 31, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.controlpoint.blocks;

import java.rmi.RemoteException;
import java.util.List;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.dsol.animation.LocatableInterface;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.animation.SwitchBlockTrafficLightAnimation;
import nl.tudelft.simulation.traffic.controlpoint.real.AbstractVisibleControlPoint;
import nl.tudelft.simulation.traffic.controlpoint.real.Changeable;
import nl.tudelft.simulation.traffic.controlpoint.real.StopSignInterface;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * 
 * This class implements the BlockTrafficLights that guards the switch
 * 
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version May 31, 2004 <br>
 * @author Elisangela Kanacilo
 */
public class SwitchBlockTrafficLight extends AbstractVisibleControlPoint
        implements
            LocatableInterface,
            StopSignInterface
{
    /** GREEN state */
    public static final String STATE_GREEN = "GREEN";

    /** RED state */
    public static final String STATE_RED = "RED";

    /** BLACK state */
    public static final String STATE_BLACK = "BLACK";

    /** REQUEST sensor */
    public static final int SENSOR_REQUEST = 0;

    /** RED sensor */
    public static final int SENSOR_RED = 1;

    /** RELEASE sensor */
    public static final int SENSOR_RELEASE = 2;

    /** sensor names */
    public static final String[] sensorNames = new String[]{"REQUEST", "RED",
            "RELEASE"};

    /**
     * the distance to the sensor BEFORE the traffic light to request access to
     * the block
     */
    private double distanceRequest;

    /** the distance to the sensor AFTER the traffic light for RED */
    private double distanceRed;

    /** the distance to the sensor AFTER the traffic light to release the block */
    private double distanceRelease;

    /** the STATE of the current block signal */
    private String currentState = STATE_RED;

    /** the name */
    private String name;

    /** dx for animation (bounds) */
    private double dx;

    /** dy for animation (bounds) */
    private double dy;

    /** the block to which the trafic light belongs */
    private VirtualBlock virtualBlock;

    /** the vehicle requesting access */
    private VehiclePhysicalInterface requestingVehicle;

    /**
     * @param name
     * @param track
     * @param progression
     * @param virtualBlock
     * @param visibleDistance
     * @param distanceRequest
     * @param distanceRed
     * @param simulator
     * @param dx
     * @param dy
     */
    public SwitchBlockTrafficLight(final String name,
            final TrackInterface track, final double progression,
            final VirtualBlock virtualBlock, final double visibleDistance,
            final double distanceRequest, final double distanceRed,
            final DEVSSimulatorInterface simulator, final double dx,
            final double dy)
    {
        super(track, progression, visibleDistance, simulator);
        this.name = name;
        System.out.println(this + ", placed on " + track + " progression "
                + progression + " (length=" + track.getLength() + ")");
        this.virtualBlock = virtualBlock;
        this.distanceRed = distanceRed;
        this.distanceRequest = distanceRequest;
        double vcpProg = distanceRequest - progression;
        this.buildRequestControlPoints(track, vcpProg, simulator);
        addSensors(super.track, progression + this.distanceRed, SENSOR_RED,
                simulator);
        this.dx = dx;
        this.dy = dy;
        if (simulator instanceof AnimatorInterface)
        {
            new SwitchBlockTrafficLightAnimation(this, simulator, dx, dy);
        }
    }

    /**
     * @param track
     * @param progression
     * @param sensor
     * @param simulator
     */
    private void addSensors(final TrackInterface track, final double progression,
            final int sensor, final DEVSSimulatorInterface simulator)
    {
        List tpList = track.calculateTrackProgressionListAll(progression);
        for (int i = 0; i < tpList.size(); i++)
        {
            TrackProgression tp = (TrackProgression) tpList.get(i);
            new SwitchBlockControlPoint(tp.getTrack(), tp.getProgression(),
                    this, sensor, simulator);
            System.out.println(this + ", sensor "
                    + SwitchBlockTrafficLight.sensorNames[sensor]
                    + " added on track " + tp.getTrack() + ", progression "
                    + tp.getProgression());
        }
    }

    /**
     * Add sensors according to the line. It is used only for adding Release
     * Sensors for the block
     * 
     * @param track
     * @param progression
     * @param sensor
     * @param line
     * @param simulator
     */
    public void addSensors(final TrackInterface track, final double progression,
            final int sensor, final String line, final DEVSSimulatorInterface simulator)
    {
        List tpList = track.calculateTrackProgressionListLine(progression, line);
        for (int i = 0; i < tpList.size(); i++)
        {
            TrackProgression tp = (TrackProgression) tpList.get(i);
            new SwitchBlockControlPoint(tp.getTrack(), tp.getProgression(),
                    this, sensor, simulator);
            System.out.println(this + ", sensor "
                    + SwitchBlockTrafficLight.sensorNames[sensor]
                    + " added on track " + tp.getTrack() + ", progression "
                    + tp.getProgression());
        }
    }

    /**
     * @param track
     * @param vcpProg
     * @param simulator
     */
    private void buildRequestControlPoints(TrackInterface track,
            double vcpProg, final DEVSSimulatorInterface simulator)
    {
        List tpList = track.calculateTrackProgressionListAll(-vcpProg);
        for (int i = 0; i < tpList.size(); i++)
        {
            TrackProgression tp = (TrackProgression) tpList.get(i);
            new SwitchBlockControlPoint(tp.getTrack(), tp.getProgression(),
                    this, SENSOR_REQUEST, simulator);
            System.out.println(this + ", sensor "
                    + SwitchBlockTrafficLight.sensorNames[SENSOR_REQUEST]
                    + " added on track " + tp.getTrack() + ", progression "
                    + tp.getProgression());
        }
    }

    /**
     * @param sensor
     * @param vehicle
     */
    public void triggerSensor(final int sensor,
            final VehiclePhysicalInterface vehicle)
    {
        Logger.fine(this, "triggerSensor", this + ", Vehicle "
                + vehicle.toString() + " triggered sensor "
                + SwitchBlockTrafficLight.sensorNames[sensor]
                + ", old state was " + this.currentState);
        System.out.println(this + ", Vehicle " + vehicle.toString()
                + " triggered sensor "
                + SwitchBlockTrafficLight.sensorNames[sensor]
                + ", old state was " + this.currentState);
        if (this.currentState == STATE_BLACK)
        {
            this.changeState(STATE_BLACK);
        } else if (this.currentState == STATE_RED)
        {
            switch (sensor)
            {
                case SENSOR_REQUEST :
                    this.requestingVehicle = vehicle;
                    this.virtualBlock.requestAccess(this);
                    break;
                case SENSOR_RED :
                    this.changeState(STATE_BLACK);
                    Logger.severe(this, "triggerSensor", "RED - RED");
                    break;
                case SENSOR_RELEASE :
                    this.virtualBlock.releaseBlock();
                    break;
                default :
                    Logger.severe(this, "triggerSensor",
                            "sensor not RED, REQUEST or RELEASE");
                    break;
            }
        } else if (this.currentState == STATE_GREEN)
        {
            switch (sensor)
            {
                case SENSOR_REQUEST :
                    this.changeState(STATE_BLACK);
                    Logger.severe(this, "triggerSensor", "GREEN - REQUEST");
                    break;
                case SENSOR_RED :
                    this.changeState(STATE_RED);
                    break;
                case SENSOR_RELEASE :
                    this.changeState(STATE_BLACK);
                    Logger.severe(this, "triggerSensor", "GREEN - RELEASE");
                    break;
                default :
                    Logger.severe(this, "triggerSensor",
                            "sensor not RED, REQUEST or RELEASE");
                    break;
            }
        } else
        {
            Logger.severe(this, "triggerSensor",
                    "SwitchBlockTrafficLight not in state BLACK, RED or GREEN");
        }
        Logger.fine(this, "triggerSensor", this + ", Vehicle "
                + vehicle.toString() + " new state is " + this.currentState);
        System.out.println(this + ", Vehicle " + vehicle.toString()
                + " new state is " + this.currentState);
    }

    /**
     * @param newState
     */
    protected void changeState(final String newState)
    {
        this.currentState = newState;
        Event event = new Event(Changeable.CHANGE_STATUS_EVENT, this, null);
        this.fireEvent(event);
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#pass(nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface)
     */
    public void pass(final VehiclePhysicalInterface vehicle)
    {
        // cancel the listener of the vehicle
        this.removeListener(vehicle, Changeable.CHANGE_STATUS_EVENT);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        DirectedPoint p = super.getTrack().getLocationOfProgression(
                super.getProgression());
        return new DirectedPoint(p.x, p.y, p.z);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds() throws RemoteException
    {
        return new BoundingBox(new Point3d(this.dx - 2.0, this.dy, 0),
                new Point3d(this.dx + 2.0, this.dy + 12.0, 0));
    }

    /**
     * @return Returns the currentState.
     */
    public String getCurrentState()
    {
        return this.currentState;
    }

    /**
     * @param state
     */
    public void setCurrentState(String state)
    {
        this.currentState = state;
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.real.Changeable#getStatus()
     */
    public String getStatus()
    {
        if ((this.currentState == STATE_RED)
                || (this.currentState == STATE_BLACK))
            return StopSignInterface.STOP;
        else
            return StopSignInterface.CONTINUE;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return "SwitchBlockTrafficLight " + this.name;
    }

    /**
     * @return Returns the virtualBlock.
     */
    public VirtualBlock getVirtualBlock()
    {
        return this.virtualBlock;
    }

    /**
     * @return Returns the distanceRed.
     */
    public double getDistanceRed()
    {
        return this.distanceRed;
    }

    /**
     * @return Returns the distanceRelease.
     */
    public double getDistanceRelease()
    {
        return this.distanceRelease;
    }

    /**
     * sets the distanceRelease. This parameter need to be changed when the
     * switch changes its direction
     * 
     * @param distanceRelease distance after the SwitchBlockTrafficLight that
     * the block will be released
     */
    public void setDistanceRelease(double distanceRelease)
    {
        this.distanceRelease = distanceRelease;
    }

    /**
     * @return Returns the distanceRequest.
     */
    public double getDistanceRequest()
    {
        return this.distanceRequest;
    }

    /**
     * @return Returns the name.
     */
    public String getName()
    {
        return this.name;
    }

    /**
     * @return Returns the requestingVehicle.
     */
    public VehiclePhysicalInterface getRequestingVehicle()
    {
        return this.requestingVehicle;
    }

    /**
     * @param requestingVehicle The requestingVehicle to set.
     */
    public void setRequestingVehicle(VehiclePhysicalInterface requestingVehicle)
    {
        this.requestingVehicle = requestingVehicle;
    }
}