/*
 * @(#)VirtualSwitchBlock.java Jun 20, 2004
 * 
 * Copyright (c) 2004 Delft University of Technology Jaffalaan 5, 2628 BX Delft,
 * the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.controlpoint.blocks;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.jstats.distributions.DistConstant;
import nl.tudelft.simulation.jstats.distributions.DistContinuous;
import nl.tudelft.simulation.jstats.streams.Java2Random;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.TrackLinkInterface;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * The VirtualSwitchBlock defines the block to be guarded including the turning
 * of a switch, depending on the 'line' attribute of the vehicleControl.
 * <p>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version Jun 20, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class VirtualSwitchBlock extends VirtualBlock
{
    /** the directions contained in the virtual block * */
    private Map directions = new HashMap();

    /** the simulator */
    private DEVSSimulatorInterface simulator;

    /** the time to trn the switch */
    private DistContinuous switchTime;

    /**
     * constucts a new VirtualSwitchBlock
     * 
     * @param name the name of the VirtualSwitchBlock
     * @param simulator
     * @param switchTime
     */
    public VirtualSwitchBlock(final String name,
            final DEVSSimulatorInterface simulator,
            final DistContinuous switchTime)
    {
        super(name);
        this.simulator = simulator;
        this.switchTime = switchTime;
        if (switchTime == null)
            this.switchTime = new DistConstant(new Java2Random(), 2.0);
    }

    /**
     * adds a list of direction to the block for each line
     * 
     * @param trafficLight guarding the block
     * @param lineList list of lines that will use the direction to be set
     * @param distanceRelease distance after the traffic light that the block
     * will be released for other vehicles
     * @param fromTrack start track to set the direction of the switch
     * @param toTrack end track to set the direction of the switch
     */
    public void addSwitchDirection(final SwitchBlockTrafficLight trafficLight,
            final List lineList, final double distanceRelease,
            final TrackInterface fromTrack, final TrackInterface toTrack)
    {
        for (int i = 0; i < lineList.size(); i++)
        {
            Direction direction = new Direction(trafficLight, (String) lineList
                    .get(i), distanceRelease, fromTrack, toTrack);
            String key = makeDirectionKey(trafficLight, (String) lineList
                    .get(i));
            System.out.println("Added directionKey = " + key);
            this.directions.put(key, direction);
            // the release distance for each traffic light is set here
            trafficLight.setDistanceRelease(distanceRelease);
            System.out.println("addSwitchDirection distanceRelease: "
                    + direction.getDistanceRelease()
                    + ", trafficlight.releaseDistance: "
                    + trafficLight.getDistanceRelease());
            //last argument 2 = SENSOR_RELEASE
            trafficLight.addSensors(trafficLight.getTrack(), trafficLight
                    .getProgression()
                    + trafficLight.getDistanceRelease(),
                    SwitchBlockTrafficLight.SENSOR_RELEASE, (String) lineList
                            .get(i), this.simulator);
            System.out.println(trafficLight.getName() + ", release distance: "
                    + direction.getDistanceRelease() + ", distance Red: "
                    + trafficLight.getDistanceRed() + ", distance request: "
                    + trafficLight.getDistanceRequest());
        }
    }

    /**
     * @param trafficLight
     * @param line
     * @return
     */
    private String makeDirectionKey(final SwitchBlockTrafficLight trafficLight,
            final String line)
    {
        return trafficLight.toString() + "@" + line;
    }

    /**
     * requestAcess process the request to access a SwitchBlock
     * 
     * @param switchBlockTrafficLight correspondent switchBlockTrafficLight that
     * request access to the block
     */
    public void requestAccess(
            final SwitchBlockTrafficLight switchBlockTrafficLight)
    {
        VehiclePhysicalInterface vehicle = switchBlockTrafficLight
                .getRequestingVehicle();
        System.out.println("VirtualSwitchBlock - request access line "
                + vehicle.getVehicleControl().getLine() + " at "
                + switchBlockTrafficLight.getName());
        if (super.busy == 0)
        {
            super.busy++;
            System.out.println("variable busy:" + super.busy);
            changeSwitchDirection(switchBlockTrafficLight, vehicle
                    .getVehicleControl().getLine());
        } else
        {
            this.requestList.add(switchBlockTrafficLight);
        }
    }

    /**
     * changeSwitchDirection changes the switch to the position of the track
     * when the position of the switch is changed, a new release distance for
     * the traffic light is set
     * 
     * @param switchBlockTrafficLight
     * @param line
     */
    private void changeSwitchDirection(
            final SwitchBlockTrafficLight switchBlockTrafficLight,
            final String line)
    {
        System.out.println(makeDirectionKey(switchBlockTrafficLight, line));
        Direction direction = (Direction) this.directions.get(makeDirectionKey(
                switchBlockTrafficLight, line));
        System.out
                .println("changing switch direction. Release distance to be activated: "
                        + direction.getDistanceRelease());
        switchBlockTrafficLight.setDistanceRelease(direction
                .getDistanceRelease());
        TrackInterface fromTrack = direction.getFromTrack();
        TrackInterface toTrack = direction.getToTrack();
        TrackLinkInterface startLink = toTrack.getStartLink();
        if (startLink.getActiveSuccessor(fromTrack).equals(toTrack))
        {
            System.out.println(startLink.toString()
                    + " Switch already on the right position");
        } else if (!startLink.getActiveSuccessor(fromTrack).equals(toTrack))
        {
            try
            {
                this.simulator.scheduleEvent(this.switchTime.draw(), this,
                        this, "turnSwitch", new Object[]{startLink, fromTrack,
                                toTrack, switchBlockTrafficLight});
            } catch (Exception e)
            {
                e.printStackTrace();
            }
        } else
        {
            switchBlockTrafficLight
                    .changeState(SwitchBlockTrafficLight.STATE_GREEN);
        }
    }

    /**
     * turnSwitch turns the switch to the position from fromTrack to toTrack
     * 
     * @param startLink start link of the switch
     * @param fromTrack track before the switch
     * @param toTrack track to where the switch needs to be positioned
     * @param switchBlockTrafficLight
     */
    protected void turnSwitch(final TrackLinkInterface startLink,
            final TrackInterface fromTrack, final TrackInterface toTrack,
            SwitchBlockTrafficLight switchBlockTrafficLight)
    {
        startLink.setActiveSuccessor(fromTrack, toTrack);
        switchBlockTrafficLight
                .changeState(SwitchBlockTrafficLight.STATE_GREEN);
    }

    /**
     * An innerclass containing the record.
     */
    private static class Direction
    {
        /** the traficLight * */
        private SwitchBlockTrafficLight trafficLight = null;

        /** the line */
        private String line;

        /** distance to relase the block */
        private double distanceRelease = 0;

        /** the track */
        private TrackInterface fromTrack = null;

        /** the track */
        private TrackInterface toTrack = null;

        /**
         * construct a new Direction
         * 
         * @param trafficLight the trafficLight
         * @param line line that follows this direction
         * @param distanceRelease distance after the switchBlockTrafficLight
         * that the block will be released
         * @param fromTrack the predecessor 'from' track
         * @param toTrack the successor 'to' track
         */
        public Direction(final SwitchBlockTrafficLight trafficLight,
                final String line, final double distanceRelease,
                final TrackInterface fromTrack, final TrackInterface toTrack)
        {
            super();
            this.trafficLight = trafficLight;
            this.distanceRelease = distanceRelease;
            this.line = line;
            this.fromTrack = fromTrack;
            this.toTrack = toTrack;
            System.out.println("Direction added: " + this.trafficLight + ", "
                    + this.distanceRelease + ", " + this.line + ", "
                    + this.fromTrack.toString() + ", "
                    + this.toTrack.toString());
        }

        /**
         * @return Returns the line.
         */
        public String getLine()
        {
            return this.line;
        }

        /**
         * @return Returns the fromTrack.
         */
        public TrackInterface getFromTrack()
        {
            return this.fromTrack;
        }

        /**
         * @return Returns the toTrack.
         */
        public TrackInterface getToTrack()
        {
            return this.toTrack;
        }

        /**
         * @return Returns the trafficLight.
         */
        public SwitchBlockTrafficLight getTrafficLight()
        {
            return this.trafficLight;
        }

        /**
         * @return distanceRelease
         */
        public double getDistanceRelease()
        {
            return this.distanceRelease;
        }
    }
}