/*
 * Created on 29-dec-2003
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.controlpoint.real;

import java.rmi.RemoteException;
import java.util.List;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.traffic.animation.ControlPointAnimation;
import nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface;
import nl.tudelft.simulation.traffic.controlpoint.util.ControlPointsList;
import nl.tudelft.simulation.traffic.controlpoint.virtual.Show;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;

/**
 * This abstract class describes the basics of a visibleControlPoint.
 * 
 * It takes care of the construction of showControlPoints. <br>
 */
public abstract class AbstractVisibleControlPoint extends EventProducer
        implements
            VisibleControlPointInterface
{
    /** the list of control points */
    private ControlPointsList showControlPoints;

    /** the track */
    protected TrackInterface track;

    /** the progression */
    protected double progression;

    /** the visible distance */
    private double visibleDistance;

    /** the location of the control point */
    private DirectedPoint location;

    /**
     * This constructor creates a new instance of the
     * AbstractVisibleControlPoint. It defines the track on which the
     * controlPoint is situated and the associated progression, as well the
     * distance from which the ControlPoint becomes visible. <br>
     * 
     * @param track
     * @param progression
     * @param visibleDistance
     * @param simulator
     */
    public AbstractVisibleControlPoint(final TrackInterface track,
            final double progression, final double visibleDistance,
            final SimulatorInterface simulator)
    {
        this.showControlPoints = new ControlPointsList();
        // as the progression may cross the track length, calculate the right
        // place (at the moment for the active successors)
        TrackProgression tp = track
                .calculateTrackProgressionListActive(progression);
        this.track = tp.getTrack();
        this.progression = tp.getProgression();
        this.location = this.track.getLocationOfProgression(this.progression);
        this.location.setRotZ(0.0);
        this.visibleDistance = visibleDistance;
        this.track.addControlPoint(this);
        double vcpProg = visibleDistance - this.progression;
        buildShowControlPoints(this.track, vcpProg, simulator);
        // create animation
        if (simulator instanceof AnimatorInterface)
        {
            new ControlPointAnimation(this, simulator);
        }
    }

    /**
     * This method creates a show for each path along which this
     * visibleControlPoint is reachable Each show is defined in the same
     * direction as the direction of this visibleControlPoint.
     * 
     * @see nl.tudelft.simulation.traffic.controlpoint.virtual.ShowInterface
     * 
     * @param track
     * @param visibleDistance
     * @param simulator
     */
    private void buildShowControlPoints(final TrackInterface track,
            final double visibleDistance, final SimulatorInterface simulator)
    {
        List tpList = track.calculateTrackProgressionListAll(-visibleDistance);
        for (int i = 0; i < tpList.size(); i++)
        {
            TrackProgression tp = (TrackProgression) tpList.get(i);
            ControlPointInterface show = new Show(tp.getTrack(), tp
                    .getProgression(), this, simulator);
            this.showControlPoints.add(show);
            System.out.println(this + ", show " + " added on track "
                    + tp.getTrack() + ", progression " + tp.getProgression());
        }
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#getProgression()
     */
    public double getProgression()
    {
        return this.progression;
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#getTrack()
     */
    public TrackInterface getTrack()
    {
        return this.track;
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.real.VisibleControlPointInterface#getVisibleDistance()
     */
    public double getVisibleDistance()
    {
        return this.visibleDistance;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds() throws RemoteException
    {
        return new BoundingBox(new Point3d(-0.1, -0.1, 0), new Point3d(0.1,
                0.1, 0));
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        return this.location;
    }
}