/*
 * Created on Dec 18, 2003
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.controlpoint.real;

import java.rmi.RemoteException;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.dsol.SimRuntimeException;
import nl.tudelft.simulation.dsol.animation.LocatableInterface;
import nl.tudelft.simulation.dsol.formalisms.devs.SimEvent;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.jstats.distributions.DistContinuous;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.traffic.animation.TrafficLightAnimation;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * This class is an implementation of a simple trafficLight. It is based on the
 * AbstractVisibleContolPoint and the StopSignInterface. The
 * LocatableInterface is only used for the animation. <br>
 * <br>
 * 
 * @author J.H. Kwakkel & H.W.G. Phaff
 * 
 * @see nl.tudelft.simulation.traffic.controlpoint.real.StopSignInterface
 * @see nl.tudelft.simulation.traffic.controlpoint.real.AbstractVisibleControlPoint
 */
public class TrafficLight extends AbstractVisibleControlPoint
        implements
            StopSignInterface,
            LocatableInterface
{
    /** the simulator to schedule on */
    private DEVSSimulatorInterface simulator;

    /** the duration of red time */
    private DistContinuous durationRed;

    /** the duration of green time */
    private DistContinuous durationGreen;

    /** current status of the traffic light */
    private String status;

    /** the name */
    private String name;

    /** dx for animation (bounds) */
    private double dx;

    /** dy for animation (bounds) */
    private double dy;

    /**
     * @param name
     * @param progression
     * @param visibleDistance
     * @param track
     * @param durationGreen
     * @param durationRed
     * @param status
     * @param simulator
     * @param dx
     * @param dy
     */
    public TrafficLight(final String name, final double progression,
            final double visibleDistance, final TrackInterface track,
            final DistContinuous durationGreen,
            final DistContinuous durationRed, final String status,
            final DEVSSimulatorInterface simulator, final double dx,
            final double dy)
    {
        super(track, progression, visibleDistance, simulator);
        this.name = name;
        System.out.println(this + ", placed on " + track + " progression "
                + progression + " (length=" + track.getLength() + ")");
        this.status = status;
        this.durationRed = durationRed;
        this.durationGreen = durationGreen;
        this.simulator = simulator;
        this.dx = dx;
        this.dy = dy;
        if (simulator instanceof AnimatorInterface)
        {
            new TrafficLightAnimation(this, simulator, dx, dy);
        }
        double timeToNext = 0;
        try
        {
            if (this.status.equals(StopSignInterface.CONTINUE))
            {
                timeToNext = this.durationGreen.draw();
            } else
            {
                timeToNext = this.durationRed.draw();
            }
            SimEvent simEvent = new SimEvent(simulator.getSimulatorTime()
                    + timeToNext, this, this, "changeStatus", null);
            simulator.scheduleEvent(simEvent);
        } catch (RemoteException e1)
        {
            e1.printStackTrace();
        } catch (SimRuntimeException e1)
        {
            e1.printStackTrace();
        }
    }

    /**
     * @throws Exception
     */
    protected void changeStatus() throws Exception
    {
        double timeToNext = 0;
        if (this.status == StopSignInterface.CONTINUE)
        {
            this.status = StopSignInterface.STOP;
            Event event = new Event(Changeable.CHANGE_STATUS_EVENT, this, null);
            this.fireEvent(event);
            timeToNext = this.durationRed.draw();
        } else
        {
            this.status = StopSignInterface.CONTINUE;
            Event event = new Event(Changeable.CHANGE_STATUS_EVENT, this, null);
            this.fireEvent(event);
            timeToNext = this.durationGreen.draw();
        }
        SimEvent simEvent = new SimEvent(this.simulator.getSimulatorTime()
                + timeToNext, this, this, "changeStatus", null);
        this.simulator.scheduleEvent(simEvent);
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#pass(nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface)
     */
    public void pass(final VehiclePhysicalInterface vehicle)
    {
        // cancel the listener of the vehicle
        this.removeListener(vehicle, Changeable.CHANGE_STATUS_EVENT);
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#getProgression()
     */
    public double getProgression()
    {
        return super.getProgression();
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.real.Changeable#getStatus()
     */
    public String getStatus()
    {
        return this.status;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        DirectedPoint p = super.getTrack().getLocationOfProgression(
                super.getProgression());
        return new DirectedPoint(p.x, p.y, p.z);
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#getTrack()
     */
    public TrackInterface getTrack()
    {
        return super.getTrack();
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds() throws RemoteException
    {
        return new BoundingBox(new Point3d(this.dx - 2.0, this.dy, 0),
                new Point3d(this.dx + 2.0, this.dy + 8.0, 0));
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return "TrafficLight " + this.name;
    }
}