/*
 * @(#) InfraFileXMLParser.java Jun 6, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.file;

import java.awt.Color;
import java.awt.geom.Point2D;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import nl.tudelft.simulation.dsol.experiment.TimeUnit;
import nl.tudelft.simulation.dsol.experiment.TimeUnitInterface;
import nl.tudelft.simulation.dsol.simulators.DEVDESSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.jstats.distributions.DistConstant;
import nl.tudelft.simulation.jstats.distributions.DistContinuous;
import nl.tudelft.simulation.jstats.distributions.DistExponential;
import nl.tudelft.simulation.jstats.distributions.DistNormal;
import nl.tudelft.simulation.jstats.distributions.DistTriangular;
import nl.tudelft.simulation.jstats.streams.Java2Random;
import nl.tudelft.simulation.jstats.streams.StreamInterface;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.language.io.URLResource;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.animation.ArcTrackAnimation;
import nl.tudelft.simulation.traffic.animation.StraightTrackAnimation;
import nl.tudelft.simulation.traffic.controlpoint.blocks.BlockTrafficLight;
import nl.tudelft.simulation.traffic.controlpoint.blocks.SwitchBlockTrafficLight;
import nl.tudelft.simulation.traffic.controlpoint.blocks.VirtualSwitchBlock;
import nl.tudelft.simulation.traffic.controlpoint.real.RemoveControlPoint;
import nl.tudelft.simulation.traffic.controlpoint.real.SpeedSign;
import nl.tudelft.simulation.traffic.controlpoint.real.StopSignInterface;
import nl.tudelft.simulation.traffic.controlpoint.real.TrafficLight;
import nl.tudelft.simulation.traffic.station.Station;
import nl.tudelft.simulation.traffic.track.ArcTrack;
import nl.tudelft.simulation.traffic.track.ComplexTrackLink;
import nl.tudelft.simulation.traffic.track.SimpleTrackLink;
import nl.tudelft.simulation.traffic.track.StraightTrack;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.TrackLinkInterface;
import nl.tudelft.simulation.traffic.track.VirtualTrack;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;
import nl.tudelft.simulation.traffic.vehicle.AccelerationProfile;
import nl.tudelft.simulation.traffic.vehicle.DecelerationProfile;
import nl.tudelft.simulation.traffic.vehicle.VehicleControl;
import nl.tudelft.simulation.traffic.vehicle.VehicleControlInterface;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysical;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;
import nl.tudelft.simulation.traffic.vehicle.VehicleType;
import nl.tudelft.simulation.xml.AbstractXMLParser;
import nl.tudelft.simulation.xml.language.ColorParser;
import nl.tudelft.simulation.xml.language.LocationParser;
import org.jdom.Element;

/**
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version May 31, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class InfraFileXMLParser extends AbstractXMLParser
{
    /** the default schema file */
    public static final URL INFRAFILE_SCHEMA = URLResource
            .getResource("/infrafile.xsd");

    /** the map of link names to link elements */
    private HashMap linkMap = new HashMap();

    /** the map of track names to track elements */
    private HashMap trackMap = new HashMap();

    /** the map of vehicle types */
    private HashMap vehicleTypeMap = new HashMap();

    /** the simulator */
    private DEVSSimulatorInterface simulator;

    /** the width of the rails in meters */
    private double railwidth = 1.48;

    /** the stream to use in random values */
    private static StreamInterface stream = new Java2Random();

    /**
     * @param url
     * @param schema
     * @param schemaNamespace
     * @param validateSchema
     * @param simulator
     * @throws Exception
     */
    public InfraFileXMLParser(final URL url, final URL schema,
            final String schemaNamespace, final boolean validateSchema,
            final DEVSSimulatorInterface simulator) throws Exception
    {
        super(url, schema, schemaNamespace, validateSchema);
        this.simulator = simulator;
        Element rootElement = parse();
        parse(rootElement);
    }

    /**
     * @param url
     * @param schema
     * @param schemaNamespace
     * @param validateSchema
     * @param simulator
     * @param linkMap
     * @param trackMap
     * @throws Exception
     */
    public InfraFileXMLParser(final URL url, final URL schema,
            final String schemaNamespace, final boolean validateSchema,
            final DEVSSimulatorInterface simulator, final HashMap linkMap,
            final HashMap trackMap) throws Exception
    {
        super(url, schema, schemaNamespace, validateSchema);
        this.simulator = simulator;
        this.linkMap = new HashMap(linkMap);
        this.trackMap = new HashMap(trackMap);
        Element rootElement = parse();
        parse(rootElement);
    }

    /**
     * parses an infrafile URL
     * 
     * @param xmlMapFileElement
     * @throws Exception on parse exception
     */
    protected void parse(final Element xmlMapFileElement) throws Exception
    {
        try
        {
            // the links
            Element links = xmlMapFileElement.getChild("links");
            if (links == null)
                System.err.println("links element not found...");
            else
            {
                java.util.List linkElements = links.getChildren("link");
                for (Iterator iterator = linkElements.iterator(); iterator
                        .hasNext();)
                {
                    Element linkElement = (Element) iterator.next();
                    parseLink(linkElement);
                }
            }
            // the tracks
            Element tracks = xmlMapFileElement.getChild("tracks");
            if (tracks == null)
                System.err.println("tracks element not found...");
            else
            {
                java.util.List trackElements = tracks.getChildren("track");
                for (Iterator iterator = trackElements.iterator(); iterator
                        .hasNext();)
                {
                    Element trackElement = (Element) iterator.next();
                    parseTrack(trackElement).toString();
                }
            }
            // the active successors
            Element successors = xmlMapFileElement.getChild("activeSuccessors");
            if (successors == null)
                System.err.println("activeSuccessors element not found...");
            else
            {
                java.util.List successorElements = successors
                        .getChildren("successor");
                for (Iterator iterator = successorElements.iterator(); iterator
                        .hasNext();)
                {
                    Element successorElement = (Element) iterator.next();
                    parseSuccessor(successorElement);
                }
            }
            //stations
            Element stations = xmlMapFileElement.getChild("stations");
            if (stations != null)
            {
                java.util.List stationElements = stations
                        .getChildren("station");
                for (Iterator iterator = stationElements.iterator(); iterator
                        .hasNext();)
                {
                    Element stationElement = (Element) iterator.next();
                    parseStation(stationElement);
                }
            }
            // the signals
            Element signals = xmlMapFileElement.getChild("signals");
            if (signals != null)
            {
                java.util.List speedSignElements = signals
                        .getChildren("speedSign");
                for (Iterator iterator = speedSignElements.iterator(); iterator
                        .hasNext();)
                {
                    Element speedSignElement = (Element) iterator.next();
                    parseSpeedSign(speedSignElement);
                }
                java.util.List trafficLightRandomElements = signals
                        .getChildren("trafficLightRandom");
                for (Iterator iterator = trafficLightRandomElements.iterator(); iterator
                        .hasNext();)
                {
                    Element trafficLightRandomElement = (Element) iterator
                            .next();
                    parseTrafficLightRandom(trafficLightRandomElement);
                }
                java.util.List blockTrafficLightElements = signals
                        .getChildren("blockTrafficLight");
                for (Iterator iterator = blockTrafficLightElements.iterator(); iterator
                        .hasNext();)
                {
                    Element blockTrafficLightElement = (Element) iterator
                            .next();
                    parseBlockTrafficLight(blockTrafficLightElement);
                }
                java.util.List switchBlockElements = signals
                        .getChildren("switchBlock");
                for (Iterator iterator = switchBlockElements.iterator(); iterator
                        .hasNext();)
                {
                    Element switchBlockElement = (Element) iterator.next();
                    parseSwitchBlock(switchBlockElement);
                }
            }
            // the vehicle types
            Element vehicleTypes = xmlMapFileElement.getChild("vehicleTypes");
            if (vehicleTypes == null)
                System.err.println("vehicleTypes element not found...");
            else
            {
                java.util.List vehicleTypesElements = vehicleTypes
                        .getChildren("vehicleType");
                for (Iterator iterator = vehicleTypesElements.iterator(); iterator
                        .hasNext();)
                {
                    Element vehicleTypeElement = (Element) iterator.next();
                    parseVehicleType(vehicleTypeElement);
                }
            }
            // the vehicles
            /*******************************************************************
             * Element vehicles = xmlMapFileElement.getChild("vehicles"); if
             * (vehicles == null) System.err.println("vehicles element not
             * found..."); else { java.util.List vehiclesElements = vehicles
             * .getChildren("vehicle"); for (Iterator iterator =
             * vehiclesElements.iterator(); iterator .hasNext();) { Element
             * vehicleElement = (Element) iterator.next();
             * parseVehicle(vehicleElement); } }
             ******************************************************************/
            // the generated vehicles
            Element genVehicles = xmlMapFileElement.getChild("genVehicles");
            if (genVehicles == null)
                System.out.println("genVehicles element not found...");
            else
            {
                java.util.List genVehiclesElements = genVehicles
                        .getChildren("genVehicle");
                for (Iterator iterator = genVehiclesElements.iterator(); iterator
                        .hasNext();)
                {
                    Element genVehicleElement = (Element) iterator.next();
                    parseGenVehicle(genVehicleElement);
                }
                java.util.List delVehiclesElements = genVehicles
                        .getChildren("delVehicle");
                for (Iterator iterator = delVehiclesElements.iterator(); iterator
                        .hasNext();)
                {
                    Element delVehicleElement = (Element) iterator.next();
                    parseDelVehicle(delVehicleElement);
                }
            }
        } catch (Exception exception)
        {
            throw exception;
        }
    }

    /**
     * parses a xml-element representing a (dx, dy) animation offset
     * 
     * @param element The j-dom element
     * @return Point2D
     * @throws IOException
     */
    private Point2D parseAnimationOffset(final Element element)
            throws IOException
    {
        try
        {
            double dx = new Double(element.getChildText("dx")).doubleValue();
            double dy = new Double(element.getChildText("dy")).doubleValue();
            return new Point2D.Double(dx, dy);
        } catch (Exception exception)
        {
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a time value with a time unit
     * 
     * @param element The j-dom element
     * @return the units on the simulator
     * @throws IOException
     */
    private double parseTimeUnit(final Element element) throws IOException
    {
        try
        {
            double value = new Double(element.getChildText("value"))
                    .doubleValue();
            String unit = element.getChildText("unit");
            if ("second".equals(unit))
                return TimeUnit.convert(value, TimeUnitInterface.SECOND,
                        this.simulator);
            else if ("minute".equals(unit))
                return TimeUnit.convert(value, TimeUnitInterface.MINUTE,
                        this.simulator);
            else if ("hour".equals(unit))
                return TimeUnit.convert(value, TimeUnitInterface.HOUR,
                        this.simulator);
            else
                throw new Exception("Unknown time unit " + unit);
        } catch (Exception exception)
        {
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a time value with a speed unit
     * 
     * @param element The j-dom element
     * @return the units on the simulator
     * @throws IOException
     */
    private double parseSpeedUnit(final Element element) throws IOException
    {
        try
        {
            double value = new Double(element.getChildText("value"))
                    .doubleValue();
            String unit = element.getChildText("unit");
            if ("mps".equals(unit))
                return value
                        / TimeUnit.convert(1, TimeUnitInterface.SECOND,
                                this.simulator);
            else if ("kmph".equals(unit))
                return 1000.0
                        * value
                        / TimeUnit.convert(1, TimeUnitInterface.HOUR,
                                this.simulator);
            else
                throw new Exception("Unknown speed unit " + unit);
        } catch (Exception exception)
        {
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a distribution function for a time
     * value
     * 
     * @param element The j-dom element
     * @return the units on the simulator
     * @throws IOException
     */
    private DistContinuous parseContinuousTimeDistribution(final Element element)
            throws IOException
    {
        try
        {
            String streamId = element.getChildText("stream");
            StreamInterface stream = this.simulator.getReplication().getStream(
                    streamId);
            if (stream == null)
                stream = new Java2Random();
            if (element.getChild("triangular") != null)
            {
                Element distribution = element.getChild("triangular");
                double a = new Double(parseTimeUnit(distribution.getChild("a")))
                        .doubleValue();
                double b = new Double(parseTimeUnit(distribution.getChild("b")))
                        .doubleValue();
                double c = new Double(parseTimeUnit(distribution.getChild("c")))
                        .doubleValue();
                return new DistTriangular(stream, a, b, c);
            } else if (element.getChild("normal") != null)
            {
                Element distribution = element.getChild("normal");
                double mean = new Double(parseTimeUnit(distribution
                        .getChild("mean"))).doubleValue();
                double stdev = new Double(parseTimeUnit(distribution
                        .getChild("stdev"))).doubleValue();
                return new DistNormal(stream, mean, stdev);
            } else if (element.getChild("exponential") != null)
            {
                Element distribution = element.getChild("exponential");
                double lambda = new Double(parseTimeUnit(distribution
                        .getChild("lambda"))).doubleValue();
                return new DistExponential(stream, lambda);
            } else if (element.getChild("constant") != null)
            {
                Element distribution = element.getChild("constant");
                double c = new Double(parseTimeUnit(distribution.getChild("c")))
                        .doubleValue();
                return new DistConstant(stream, c);
            } else
                throw new Exception("Unknown distribution in "
                        + element.getName());
        } catch (Exception exception)
        {
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a Link
     * 
     * @param element The j-dom element
     * @return the Link
     * @throws IOException
     */
    private TrackLinkInterface parseLink(final Element element)
            throws IOException
    {
        DirectedPoint location = new DirectedPoint();
        String name = "";
        try
        {
            name = element.getChildText("name");
            location = LocationParser.parseLocation(element
                    .getChild("location"));
            boolean simple = "simple".equals(element.getChildText("type"));
            TrackLinkInterface link;
            if (simple)
                link = new SimpleTrackLink(name, location);
            else
                link = new ComplexTrackLink(name, location);
            this.linkMap.put(name, link);
            Logger.finer(this, "parseLink", link.toString());
            return link;
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a Track
     * 
     * @param element The j-dom element
     * @return Track of element
     * @throws IOException
     */
    private TrackInterface parseTrack(final Element element) throws IOException
    {
        try
        {
            String name = element.getChildText("name");
            String startLinkString = element.getChildText("startLink");
            TrackLinkInterface startLink = (TrackLinkInterface) this.linkMap
                    .get(startLinkString);
            if (startLink == null)
            {
                throw new Exception("Track " + name + ": start link "
                        + startLinkString + " not defined");
            }
            String endLinkString = element.getChildText("endLink");
            TrackLinkInterface endLink = (TrackLinkInterface) this.linkMap
                    .get(endLinkString);
            if (endLink == null)
            {
                throw new Exception("Track " + name + ": end link "
                        + endLinkString + " not defined");
            }
            String type = element.getChildText("type");
            if ("straight".equals(type))
            {
                TrackInterface track = new StraightTrack(name, startLink,
                        endLink);
                new StraightTrackAnimation(track, this.simulator,
                        this.railwidth);
                this.trackMap.put(name, track);
                Logger.finer(this, "parseTrack[straight]", track.toString());
                return track;
            } else if ("arc".equals(type))
            {
                double radius = new Double(element.getChildText("radius"))
                        .doubleValue();
                ArcTrack track = new ArcTrack(name, startLink, endLink, radius);
                new ArcTrackAnimation(track, this.simulator, this.railwidth);
                this.trackMap.put(name, track);
                Logger.finer(this, "parseTrack[arc]", track.toString());
                return track;
            } else if ("virtual".equals(type))
            {
                TrackInterface track = new VirtualTrack(name, startLink,
                        endLink);
                this.trackMap.put(name, track);
                Logger.finer(this, "parseTrack[virtual]", track.toString());
                return track;
            } else
            {
                throw new Exception("Track " + name + ": type " + type
                        + " not supported yet");
            }
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a Successor
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseSuccessor(final Element element) throws IOException
    {
        try
        {
            String linkString = element.getChildText("link");
            TrackLinkInterface link = (TrackLinkInterface) this.linkMap
                    .get(linkString);
            if (link == null)
            {
                throw new Exception("Successor: link " + link + " not defined");
            }
            String startTrackString = element.getChildText("startTrack");
            TrackInterface startTrack = (TrackInterface) this.trackMap
                    .get(startTrackString);
            if (startTrack == null)
            {
                throw new Exception("Successor: start track "
                        + startTrackString + " not defined");
            }
            if (!link.equals(startTrack.getEndLink()))
            {
                throw new Exception("Successor: link " + link
                        + " not the end link of track " + startTrackString);
            }
            String nextTrackString = element.getChildText("nextTrack");
            TrackInterface nextTrack = (TrackInterface) this.trackMap
                    .get(nextTrackString);
            if (nextTrack == null)
            {
                throw new Exception("Successor: next track " + nextTrackString
                        + " not defined");
            }
            if (!link.equals(nextTrack.getStartLink()))
            {
                throw new Exception("Successor: link " + link
                        + " not the start link of track " + nextTrackString);
            }
            boolean active = Boolean.valueOf(element.getChildText("active"))
                    .booleanValue();
            startTrack.getEndLink().setActiveSuccessor(startTrack, nextTrack);
            Logger.finer(this, "parseSuccessor", startTrack.toString() + "-"
                    + nextTrack.toString());
            if (active)
            {
                link.setActiveSuccessor(startTrack, nextTrack);
            }
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * @param track
     * @param progression
     * @return the TrackProgression
     */
    private TrackProgression correctedTrackProgression(
            final TrackInterface track, final double progression)
    {
        List tpl = track.calculateTrackProgressionListAll(progression);
        if (tpl.size() == 1)
        {
            return (TrackProgression) tpl.get(0);
        }
        Logger.severe(this, "correctedTrackProgression",
                "not one corrected location for track " + track
                        + ", progression " + progression);
        System.out.println();
        return new TrackProgression(track, progression);
    }

    /**
     * @param element
     * @throws IOException
     */
    private void parseStation(final Element element) throws IOException
    {
        try
        {
            String stationName = element.getChildText("stationName");
            String stationCode = element.getChildText("stationCode");
            String stationCompanyCode = element
                    .getChildText("stationCompanyCode");
            String startTrackString = element.getChildText("startTrack");
            TrackInterface startTrack = (TrackInterface) this.trackMap
                    .get(startTrackString);
            if (startTrack == null)
            {
                throw new Exception("Station: start track " + startTrackString
                        + "not defined");
            }
            double stationProgression = new Double(element
                    .getChildText("progression")).doubleValue();
            TrackProgression tp = correctedTrackProgression(startTrack,
                    stationProgression);
            double requestDistance = new Double(element
                    .getChildText("requestDistance")).doubleValue();
            double stationLength = new Double(element
                    .getChildText("stationLength")).doubleValue();
            double stoppingPlaceLength = new Double(element
                    .getChildText("stoppingPlaceLength")).doubleValue();
            double haltingTime = new Double(element.getChildText("haltingTime"))
                    .doubleValue();
            String platformSide = element.getChildText("platformSide");
            new Station(stationName, stationCode, stationCompanyCode, tp
                    .getTrack(), tp.getProgression(), requestDistance,
                    stationLength, stoppingPlaceLength, TimeUnit.convert(
                            haltingTime, TimeUnitInterface.SECOND,
                            this.simulator), platformSide, this.simulator);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a SpeedSign
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseSpeedSign(final Element element) throws IOException
    {
        try
        {
            String name = element.getChildText("name");
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseSpeedSign : track " + trackString
                        + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            double visibleDistance = new Double(element
                    .getChildText("visibleDistance")).doubleValue();
            Element speedLimit = element.getChild("speedLimit");
            double speedLimitValue = parseSpeedUnit(speedLimit);
            String speedLimitText = speedLimit.getChildText("text");
            Point2D animationOffset = parseAnimationOffset(element
                    .getChild("animationOffset"));
            Color backgroundColor = ColorParser.parseColor(element
                    .getChild("backgroundColor"));
            Color textColor = ColorParser.parseColor(element
                    .getChild("textColor"));
            Color lineColor = ColorParser.parseColor(element
                    .getChild("lineColor"));
            new SpeedSign(name, tp.getTrack(), tp.getProgression(),
                    visibleDistance, speedLimitValue, this.simulator,
                    animationOffset.getX(), animationOffset.getY(),
                    speedLimitText, backgroundColor, textColor, lineColor);
            Logger.finer(this, "parseSpeedSign", track.toString() + ":" + name);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a TrafficLightRandom
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseTrafficLightRandom(final Element element)
            throws IOException
    {
        try
        {
            String name = element.getChildText("name");
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseTrafficLightRandom : track "
                        + trackString + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            double visibleDistance = new Double(element
                    .getChildText("visibleDistance")).doubleValue();
            double timeGreen = new Double(element.getChildText("timeGreen"))
                    .doubleValue();
            double timeRed = new Double(element.getChildText("timeRed"))
                    .doubleValue();
            boolean initialGreen = Boolean.valueOf(
                    element.getChildText("initialGreen")).booleanValue();
            String status;
            if (initialGreen)
                status = StopSignInterface.CONTINUE;
            else
                status = StopSignInterface.STOP;
            Point2D animationOffset = parseAnimationOffset(element
                    .getChild("animationOffset"));
            new TrafficLight(name, tp.getProgression(), visibleDistance, tp
                    .getTrack(), new DistExponential(stream, timeGreen),
                    new DistExponential(stream, timeRed), status,
                    this.simulator, animationOffset.getX(), animationOffset
                            .getY());
            Logger.finer(this, "parseTrafficLightRandom", track.toString()
                    + ":" + name);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a AbsoluteBlockTrafficLight
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseBlockTrafficLight(final Element element)
            throws IOException
    {
        try
        {
            String name = element.getChildText("name");
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseBlockTrafficLight : track "
                        + trackString + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            double visibleDistance = new Double(element
                    .getChildText("visibleDistance")).doubleValue();
            double distanceRed = new Double(element.getChildText("distanceRed"))
                    .doubleValue();
            double distanceYellow = new Double(element
                    .getChildText("distanceYellow")).doubleValue();
            double distanceGreen = new Double(element
                    .getChildText("distanceGreen")).doubleValue();
            Point2D animationOffset = parseAnimationOffset(element
                    .getChild("animationOffset"));
            new BlockTrafficLight(name, tp.getTrack(), tp.getProgression(),
                    visibleDistance, distanceRed, distanceYellow,
                    distanceGreen, this.simulator, animationOffset.getX(),
                    animationOffset.getY());
            Logger.finer(this, "parseBlockTrafficLight", track.toString() + ":"
                    + name);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a VirtualSwitchBlock with
     * SwitchBlockTrafficLights
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseSwitchBlock(final Element element) throws IOException
    {
        try
        {
            String name = element.getChildText("name");
            DistContinuous switchTime = parseContinuousTimeDistribution(element
                    .getChild("switchTime"));
            System.out.println("switchBlock = " + name + " switchTime = "
                    + switchTime);
            VirtualSwitchBlock switchBlock = new VirtualSwitchBlock(name,
                    this.simulator, switchTime);
            java.util.List switchBlockTrafficLightElements = element
                    .getChildren("switchBlockTrafficLight");
            for (Iterator iterator = switchBlockTrafficLightElements.iterator(); iterator
                    .hasNext();)
            {
                Element switchBlockTrafficLightElement = (Element) iterator
                        .next();
                parseSwitchBlockTrafficLight(switchBlock,
                        switchBlockTrafficLightElement);
            }
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a AbsoluteBlockTrafficLight
     * 
     * @param switchBlock
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseSwitchBlockTrafficLight(
            final VirtualSwitchBlock switchBlock, final Element element)
            throws IOException
    {
        try
        {
            String name = element.getChildText("name");
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseSwitchBlockTrafficLight : track "
                        + trackString + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            double visibleDistance = new Double(element
                    .getChildText("visibleDistance")).doubleValue();
            double distanceRequest = new Double(element
                    .getChildText("distanceRequest")).doubleValue();
            double distanceRed = new Double(element.getChildText("distanceRed"))
                    .doubleValue();
            Point2D animationOffset = parseAnimationOffset(element
                    .getChild("animationOffset"));
            SwitchBlockTrafficLight sbtl = new SwitchBlockTrafficLight(name, tp
                    .getTrack(), tp.getProgression(), switchBlock,
                    visibleDistance, distanceRequest, distanceRed,
                    this.simulator, animationOffset.getX(), animationOffset
                            .getY());
            // get the switch settings - if needed
            Element switchSettings = element.getChild("switchSettings");
            java.util.List switchSettingsElements = switchSettings
                    .getChildren("switchSetting");
            if (switchSettingsElements != null)
            {
                for (Iterator iterator = switchSettingsElements.iterator(); iterator
                        .hasNext();)
                {
                    Element switchSettingElement = (Element) iterator.next();
                    parseSwitchSetting(sbtl, switchSettingElement);
                }
            }
            Logger.finer(this, "parseSwitchBlockTrafficLight", track.toString()
                    + ":" + name);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a SwitchSetting within a
     * AbsoluteBlockTrafficLight
     * 
     * @param sbtl the AbsoluteBlockTrafficLight
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseSwitchSetting(final SwitchBlockTrafficLight sbtl,
            final Element element) throws IOException
    {
        try
        {
            Element linesElement = element.getChild("lines");
            List lineElementList = linesElement.getChildren("line");
            List lineList = new ArrayList();
            if (lineElementList != null)
            {
                for (Iterator iterator = lineElementList.iterator(); iterator
                        .hasNext();)
                {
                    String lineText = ((Element) iterator.next()).getText();
                    System.out.println("parser line: " + lineText);
                    lineList.add(lineText);
                }
            }
            Element successor = element.getChild("successor");
            String fromTrackStr = successor.getChildText("fromTrack");
            TrackInterface fromTrack = (TrackInterface) this.trackMap
                    .get(fromTrackStr);
            if (fromTrack == null)
            {
                throw new Exception("parseSwitchSetting: fromTrack "
                        + fromTrackStr + " not defined");
            }
            TrackLinkInterface link = fromTrack.getEndLink();
            String toTrackStr = successor.getChildText("toTrack");
            TrackInterface toTrack = (TrackInterface) this.trackMap
                    .get(toTrackStr);
            if (toTrack == null)
            {
                throw new Exception("parseSwitchSetting: toTrack " + toTrackStr
                        + " not defined");
            }
            if (!link.equals(toTrack.getStartLink()))
            {
                throw new Exception("parseSwitchSetting: link " + link
                        + " not the start link of track " + toTrackStr);
            }
            double distanceRelease = new Double(element
                    .getChildText("distanceRelease")).doubleValue();
            System.out.println("parser distanceRelease: " + distanceRelease);
            ((VirtualSwitchBlock) sbtl.getVirtualBlock()).addSwitchDirection(
                    sbtl, lineList, distanceRelease, fromTrack, toTrack);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a VehicleType
     * 
     * @param element The j-dom element
     * @return Layer of element
     * @throws IOException
     */
    private VehicleType parseVehicleType(final Element element)
            throws IOException
    {
        String name = "";
        try
        {
            VehicleType vehicleType = new VehicleType();
            name = element.getChildText("name");
            vehicleType.setName(name);
            vehicleType.setLength(new Double(element.getChildText("length"))
                    .doubleValue());
            vehicleType.setWidth(new Double(element.getChildText("width"))
                    .doubleValue());
            double maxSpeed = new Double(element.getChildText("maxSpeed"))
                    .doubleValue();
            vehicleType.setVehicleMaximumSpeed(maxSpeed);
            // the acceleration profile
            AccelerationProfile accProfile = new AccelerationProfile(maxSpeed);
            java.util.List accElements = element.getChildren("acceleration");
            for (Iterator iterator = accElements.iterator(); iterator.hasNext();)
            {
                Element accElement = (Element) iterator.next();
                double speed = new Double(accElement.getChildText("speed"))
                        .doubleValue();
                double acc = new Double(accElement.getChildText("acc"))
                        .doubleValue();
                accProfile.setAccelerationFromSpeed(speed, acc);
            }
            vehicleType.setAccelerationProfile(accProfile);
            // the deceleration profile
            DecelerationProfile decProfile = new DecelerationProfile(maxSpeed);
            java.util.List decElements = element.getChildren("deceleration");
            for (Iterator iterator = decElements.iterator(); iterator.hasNext();)
            {
                Element decElement = (Element) iterator.next();
                double speed = new Double(decElement.getChildText("speed"))
                        .doubleValue();
                double dec = new Double(decElement.getChildText("dec"))
                        .doubleValue();
                decProfile.setDecelerationFromSpeed(speed, dec);
            }
            vehicleType.setDecelerationProfile(decProfile);
            //
            vehicleType.setColor(ColorParser.parseColor(element
                    .getChild("color")));
            // the segments -- if present
            Element segments = element.getChild("segments");
            if (segments != null)
            {
                java.util.List segmentsElements = segments
                        .getChildren("segment");
                for (Iterator iterator = segmentsElements.iterator(); iterator
                        .hasNext();)
                {
                    Element segmentElement = (Element) iterator.next();
                    double front = new Double(segmentElement
                            .getChildText("front")).doubleValue();
                    double axle1 = new Double(segmentElement
                            .getChildText("axle1")).doubleValue();
                    double axle2 = new Double(segmentElement
                            .getChildText("axle2")).doubleValue();
                    double length = new Double(segmentElement
                            .getChildText("length")).doubleValue();
                    vehicleType.addSegment(front, axle1, axle2, length);
                }
            }
            this.vehicleTypeMap.put(name, vehicleType);
            Logger.finer(this, "parseVehicleType", vehicleType.getName());
            System.out.println("parseVehicleType " + vehicleType.getName());
            return vehicleType;
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a Vehicle instance
     * 
     * @param element The j-dom element
     * @return Vehicle instance
     * @throws IOException
     */
    private VehicleControlInterface parseVehicle(final Element element)
            throws IOException
    {
        try
        {
            String type = element.getChildText("type");
            VehicleType vehicleType = (VehicleType) this.vehicleTypeMap
                    .get(type);
            if (vehicleType == null)
            {
                throw new Exception("parseVehicle : vehicleType " + type
                        + " not defined");
            }
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseVehicle : track " + trackString
                        + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            String line = element.getChildText("line");
            double maxSpeedStart = new Double(element
                    .getChildText("maxSpeedStart")).doubleValue();
            Logger.finer(this, "parseVehicle", vehicleType.getName()
                    + " track " + track);
            System.out.println("parseVehicle" + vehicleType.getName()
                    + " track " + track);
            VehicleControlInterface vehicleControl = new VehicleControl(line);
            VehiclePhysicalInterface vehiclePhysical = new VehiclePhysical(
                    vehicleType, vehicleControl, tp.getTrack(), tp
                            .getProgression(), maxSpeedStart,
                    (DEVDESSSimulatorInterface) this.simulator);
            vehiclePhysical.setVehicleControl(vehicleControl);
            vehicleControl.setVehiclePhysical(vehiclePhysical);
            // schedule driving at t=0
            this.simulator.scheduleEvent(0.0, this, vehicleControl,
                    "startDriving", new Object[]{});
            return vehicleControl;
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a Vehicle generator
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseGenVehicle(final Element element) throws IOException
    {
        try
        {
            String type = element.getChildText("type");
            VehicleType vehicleType = (VehicleType) this.vehicleTypeMap
                    .get(type);
            if (vehicleType == null)
            {
                throw new Exception("parseGenVehicle : vehicleType " + type
                        + " not defined");
            }
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseGenVehicle : track " + trackString
                        + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            double firstGen = parseTimeUnit(element.getChild("firstGen"));
            double interval = parseTimeUnit(element.getChild("interval"));
            String line = element.getChildText("line");
            double maxSpeedStart = new Double(element
                    .getChildText("maxSpeedStart")).doubleValue();
            Logger.finer(this, "parseGenVehicle", vehicleType.getName()
                    + " track " + tp);
            System.out.println("parseGenVehicle" + vehicleType.getName()
                    + " track " + tp);
            Object[] args = new Object[]{tp.getTrack(),
                    new Double(tp.getProgression()), new Double(maxSpeedStart),
                    vehicleType, new Double(interval), line};
            this.simulator.scheduleEvent(firstGen, this, this, "genVehicle",
                    args);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }

    /**
     * @param track
     * @param progression
     * @param maxSpeedStart
     * @param vehicleType
     * @param interval
     * @param line
     * @throws Exception
     */
    protected void genVehicle(final TrackInterface track,
            final double progression, final double maxSpeedStart,
            final VehicleType vehicleType, double interval, String line)
            throws Exception
    {
        try
        {
            VehicleControlInterface vehicleControl = new VehicleControl(line);
            VehiclePhysicalInterface vehiclePhysical = new VehiclePhysical(
                    vehicleType, vehicleControl, track, progression,
                    maxSpeedStart, (DEVDESSSimulatorInterface) this.simulator);
            vehicleControl.setVehiclePhysical(vehiclePhysical);
            vehicleControl.startDriving();
            Object[] args = new Object[]{track, new Double(progression),
                    new Double(maxSpeedStart), vehicleType,
                    new Double(interval), line};
            this.simulator.scheduleEvent(interval, this, this, "genVehicle",
                    args);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new Exception(exception.getMessage());
        }
    }

    /**
     * parses a xml-element representing a Vehicle destroyer
     * 
     * @param element The j-dom element
     * @throws IOException
     */
    private void parseDelVehicle(final Element element) throws IOException
    {
        try
        {
            String trackString = element.getChildText("track");
            TrackInterface track = (TrackInterface) this.trackMap
                    .get(trackString);
            if (track == null)
            {
                throw new Exception("parseDelVehicle : track " + trackString
                        + " not defined");
            }
            double progression = new Double(element.getChildText("progression"))
                    .doubleValue();
            TrackProgression tp = correctedTrackProgression(track, progression);
            new RemoveControlPoint(tp.getTrack(), tp.getProgression(),
                    this.simulator);
        } catch (Exception exception)
        {
            exception.printStackTrace();
            throw new IOException(exception.getMessage());
        }
    }
}