/*
 * Created on Aug 20, 2004
 * 
 * (c) copyright 2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br> See for project
 * information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br> License of use: <a
 * href="http://www.gnu.org/copyleft/gpl.html">General Public License (GPL)
 * </a>, no warranty <br>
 * 
 * @version May 31, 2004 <br> @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/elisangelak/index.htm">Elisangela
 * Mieko Kanacilo * </a>
 */

package nl.tudelft.simulation.traffic.station;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;

import nl.tudelft.simulation.dsol.animation.LocatableInterface;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.traffic.animation.StationAnimation;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;

/**
 * <br>
 * This class is the implementation of a station. A station is considered as
 * just one side of the track. Therefore, it might be that two stations have the
 * same name, but they will have different code/codecompany
 * 
 * (c) copyright * 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version Aug 20, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/elisangelak/index.htm">Elisangela
 * Kanacilo </a>
 */
public class Station implements LocatableInterface
{
    /** name of the station */
    private String name;

    /** string or number that identifies the stationTrack */
    private String stationCode;

    /** number that identifies the stationTrack */
    private String stationCompanyCode;

    /** starting track of the stationTrack */
    private TrackInterface startTrack;

    /** starting point of the station track = progression related to startTrack */
    private double progression;

    /** stationLength of the station */
    private double stationLength;

    /** distance to request access to the station */
    private double requestDistance;

    /** List of stopping places */
    private List stoppingPlacesList = new ArrayList();

    /** Side of the station */
    public static final String PLATFORM_LEFT = "LEFT";

    /** Side of the station */
    public static final String PLATFORM_RIGHT = "RIGHT";

    /** side where the station will be placed */
    public String platformSide;

    /** the standard halting time */
    private double haltingTime;

    /** the location */
    private DirectedPoint location = new DirectedPoint();

    /**
     * @param name of the station
     * @param startTrack the first track of the station (first position where
     * the vehicles will stop)
     * @param stationCode string or number that identifies the stationTrack
     * @param stationCompanyCode number that identifies the stationTrack
     * @param platformSide side where the station will be placed
     * @param stationProgression position related to the startTrack where the
     * station starts
     * @param requestDistance
     * @param stationLength stationLength of the stationTrack
     * @param stoppingPlaceLength stationLength of each stoppingPlace
     * @param haltingTime
     * @param simulator
     */
    public Station(final String name, final String stationCode,
            final String stationCompanyCode, final TrackInterface startTrack,
            final double stationProgression, final double requestDistance,
            final double stationLength, final double stoppingPlaceLength,
            final double haltingTime, final String platformSide,
            final DEVSSimulatorInterface simulator)
    {
        this.name = name;
        this.stationCode = stationCode;
        this.stationCompanyCode = stationCompanyCode;
        // as the progression may cross the track length, calculate the right
        // place (at the moment for the active successors)
        TrackProgression tp = startTrack
                .calculateTrackProgressionListActive(stationProgression);
        if (tp != null)
        {
            this.startTrack = tp.getTrack();
            this.progression = tp.getProgression();
        } else
        {
            System.err.println("NO TRACK / PROGRESSION FOR STATION " + name
                    + ", track: " + startTrack + "." + stationProgression);
            return;
        }
        this.requestDistance = requestDistance;
        this.stationLength = stationLength;
        this.haltingTime = haltingTime;
        this.platformSide = platformSide;
        //calculating all possible stopping places according to their
        // length and the stationLength
        double startLength = 0;
        while (startLength < stationLength)
        {
            StoppingPlace stoppingPlace = new StoppingPlace(this,
                    this.startTrack, this.progression - startLength,
                    this.requestDistance, stoppingPlaceLength, simulator);
            this.addStoppingPlace(stoppingPlace);
            startLength += stoppingPlaceLength;
        }
        // create the location
        Point3d p = this.startTrack.getLocationOfProgression(this.progression);
        Point3d q = this.startTrack.getLocationOfProgression(this.progression
                - this.stationLength);
        double dx = q.x - p.x;
        double dy = q.y - p.y;
        double alpha = Math.atan2(dy, dx);
        double beta;
        if (this.platformSide.equals(Station.PLATFORM_LEFT))
            beta = alpha - Math.PI / 2.0d;
        else
            beta = alpha + Math.PI / 2.0d;
        double d = 3.0d;
        this.location = new DirectedPoint(p.x + d * Math.cos(beta), p.y + d
                * Math.sin(beta), p.z, 0.0d, 0.0d, alpha);
        if (simulator instanceof AnimatorInterface)
        {
            new StationAnimation(this, simulator);
        }
    }

    /**
     * retrieve the accessible stopping place at the stationTrack
     * 
     * @param stoppingPlacesList
     * @return accessible stopping place or null if they are all occupied
     */
    public StoppingPlace getAccessibleStoppingPlace(List stoppingPlacesList)
    {
        StoppingPlace place;
        boolean found = false;
        int i = stoppingPlacesList.size() - 1;
        while (i >= 0 && !found)
        {
            place = (StoppingPlace) stoppingPlacesList.get(i);
            if (place.isClaimed())
            {
                found = true;
                break;
            }
            i--;
        }
        if (i == stoppingPlacesList.size() - 1 && found == true)
            return null;
        else
            return (StoppingPlace) stoppingPlacesList.get(i++);
    }

    /**
     * @param stoppingPlace
     */
    private void addStoppingPlace(StoppingPlace stoppingPlace)
    {
        this.stoppingPlacesList.add(stoppingPlace);
    }

    /**
     * @return Returns the station.
     */
    public String getName()
    {
        return this.name;
    }

    /**
     * @return Returns the stationLength.
     */
    public double getStationLength()
    {
        return this.stationLength;
    }

    /**
     * @return Returns the progression.
     */
    public double getProgression()
    {
        return this.progression;
    }

    /**
     * @return Returns the startTrack.
     */
    public TrackInterface getStartTrack()
    {
        return this.startTrack;
    }

    /**
     * @return Returns the stoppingPlacesList.
     */
    public List getStoppingPlacesList()
    {
        return this.stoppingPlacesList;
    }

    /**
     * @return Returns the stationCode.
     */
    public String getStationCode()
    {
        return this.stationCode;
    }

    /**
     * @return Returns the stationCompanyCode.
     */
    public String getStationCompanyCode()
    {
        return this.stationCompanyCode;
    }

    /**
     * @return Returns the platformSide.
     */
    public String getPlatformSide()
    {
        return this.platformSide;
    }

    /**
     * @return Returns the haltingTime.
     */
    public double getHaltingTime()
    {
        return this.haltingTime;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        return this.location;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds() throws RemoteException
    {
        Point3d p = this.location;
        Point3d q = this.startTrack.getLocationOfProgression(this.progression
                + this.stationLength);
        return new BoundingBox(new Point3d(0, 0, 0), new Point3d(q.x - p.x, q.y
                - p.y, 0));
    }

}