
package nl.tudelft.simulation.traffic.station;

import java.rmi.RemoteException;
import java.util.List;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.dsol.animation.LocatableInterface;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.event.Event;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.animation.StationLightAnimation;
import nl.tudelft.simulation.traffic.controlpoint.real.AbstractVisibleControlPoint;
import nl.tudelft.simulation.traffic.controlpoint.real.Changeable;
import nl.tudelft.simulation.traffic.controlpoint.real.StopSignInterface;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * This class implements the traffic light for the stations. The traffic light
 * has the objective of controlling the behavior of the vehicles at the
 * stations. They have only two states: - red -> to make the vehicles stop at
 * the station - green -> to allow them to depart (after the halting time has
 * passed <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version May 31, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class StationHaltingTrafficLight extends AbstractVisibleControlPoint
        implements
            LocatableInterface,
            StopSignInterface
{
    /** GREEN state */
    public static final String STATE_GREEN = "GREEN";

    /** RED state */
    public static final String STATE_RED = "RED";

    /** GREEN sensor */
    public static final int SENSOR_RED = 0;

    /** RED sensor */
    public static final int SENSOR_TIMER = 1;

    /** sensor names */
    public static final String[] sensorNames = new String[]{"RED", "TIMER"};

    /** the distance before the stopping place to request */
    public double requestDistance;

    /** the STATE of the current block signal */
    private String currentState = STATE_GREEN;

    /** the name */
    private String name;

    /** dx for animation (bounds) */
    private double dx;

    /** dy for animation (bounds) */
    private double dy;

    /** the station */
    public Station station;

    /** the simulator */
    public DEVSSimulatorInterface simulator;

    /**
     * @param station
     * @param name
     * @param track
     * @param progression
     * @param visibleDistance
     * @param requestDistance
     * @param simulator
     * @param dx
     * @param dy
     */
    public StationHaltingTrafficLight(final Station station, final String name,
            final TrackInterface track, final double progression,
            final double visibleDistance, final double requestDistance,
            final DEVSSimulatorInterface simulator, final double dx,
            final double dy)
    {
        super(track, progression, visibleDistance, simulator);
        this.name = name;
        this.station = station;
        System.out.println(this + ", placed on " + super.track
                + " progression " + super.progression + " (length="
                + super.track.getLength() + ")");
        this.requestDistance = requestDistance;
        double vcpProg = progression - requestDistance;
        addStationSensors(super.track, vcpProg, SENSOR_RED, simulator);
        vcpProg = progression - 1;
        addStationSensors(super.track, vcpProg, SENSOR_TIMER, simulator);
        this.simulator = simulator;
        this.dx = dx;
        this.dy = dy;
        if (simulator instanceof AnimatorInterface)
        {
            new StationLightAnimation(this, simulator, dx, dy);
        }
    }

    /**
     * This method creates a sensor for each path along which this ControlPoint
     * is reachable, also when there are branches in the path
     * 
     * @param track
     * @param vcpProg
     * @param sensor
     * @param simulator
     */
    private void addStationSensors(final TrackInterface track,
            final double vcpProg, final int sensor,
            final SimulatorInterface simulator)
    {
        List tpList = track.calculateTrackProgressionListAll(vcpProg);
        for (int i = 0; i < tpList.size(); i++)
        {
            TrackProgression tp = (TrackProgression) tpList.get(i);
            new StationControlPoint(tp.getTrack(), tp.getProgression(), this,
                    sensor, simulator);
            System.out.println(this + ", sensor "
                    + StationHaltingTrafficLight.sensorNames[sensor]
                    + " added on track " + tp.getTrack() + ", progression "
                    + tp.getProgression());
        }
    }

    /**
     * @param sensor
     * @param vehicle
     */
    public void triggerSensor(final int sensor,
            final VehiclePhysicalInterface vehicle)
    {
        Logger.fine(this, "triggerSensor", this + ", Vehicle "
                + vehicle.toString() + " triggered sensor "
                + StationHaltingTrafficLight.sensorNames[sensor]
                + ", old state was " + this.currentState);
        System.out.println(this + ", Vehicle " + vehicle.toString()
                + " triggered sensor "
                + StationHaltingTrafficLight.sensorNames[sensor]
                + ", old state was " + this.currentState);
        switch (sensor)
        {
            case SENSOR_RED :
                this.currentState = STATE_RED;
                break;
            case SENSOR_TIMER :
                System.out.println("station.getHaltingTime: "
                        + (this.station.getHaltingTime()));
                try
                {
                    this.simulator.scheduleEvent(this.station.getHaltingTime(),
                            this, this, "stateGreen", null);
                } catch (Exception e)
                {
                    Logger.severe(this, "triggerSensor", e);
                }
                break;
            default :
                Logger.severe(this, "triggerSensor",
                        "reported state not RED or TIMER");
                break;
        }
        Logger.fine(this, "triggerSensor", this + ", Vehicle "
                + vehicle.toString() + " new state is " + this.currentState);
        System.out.println(this + ", Vehicle " + vehicle.toString()
                + " new state is " + this.currentState);
        Event event = new Event(Changeable.CHANGE_STATUS_EVENT, this, null);
        this.fireEvent(event);
    }

    /**
     *  
     */
    public void stateGreen()
    {
        System.out.println("TIMER state turned to GREEN");
        this.currentState = STATE_GREEN;
        Event event = new Event(Changeable.CHANGE_STATUS_EVENT, this, null);
        this.fireEvent(event);
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface#pass(nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface)
     */
    public void pass(final VehiclePhysicalInterface vehicle)
    {
        // cancel the listener of the vehicle
        this.removeListener(vehicle, Changeable.CHANGE_STATUS_EVENT);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        DirectedPoint p = super.getTrack().getLocationOfProgression(
                super.getProgression());
        return new DirectedPoint(p.x, p.y, p.z);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds() throws RemoteException
    {
        return new BoundingBox(new Point3d(this.dx - 2.0, this.dy, 0),
                new Point3d(this.dx + 2.0, this.dy + 12.0, 0));
    }

    /**
     * @return Returns the currentState.
     */
    public String getCurrentState()
    {
        return this.currentState;
    }

    /**
     * @see nl.tudelft.simulation.traffic.controlpoint.real.Changeable#getStatus()
     */
    public String getStatus()
    {
        if (this.currentState == STATE_RED)
            return StopSignInterface.STOP;
        else
            return StopSignInterface.CONTINUE;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return "StationHaltingTrafficLight " + this.name;
    }

    /**
     * @return
     */
    public Station getStation()
    {
        return this.station;
    }
}