/*
 * @(#)StraightTrack.java May 30, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.track;

import java.awt.geom.Point2D;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.language.d2.Angle;
import nl.tudelft.simulation.language.d2.Circle;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;

/**
 * A curved track from one link to another link. <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version May 30, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class ArcTrack extends Track
{
    /** the length of the track */
    private double length;

    /** the angle of the dx/dy of the track */
    double theta = 0.0;

    /** the radius of the arc */
    private double radius;

    /** the center point of the arc */
    private Point2D center;

    /** the start angle of the arc */
    private double startAngle;

    /** the end angle of the arc */
    private double endAngle;

    /** the delta-angle of the arc */
    private double deltaAngle;

    /** counter clockwise movement? */
    private boolean counterClockwise;

    /**
     * @param name
     * @param startLink
     * @param endLink
     * @param radius
     * @throws Exception
     */
    public ArcTrack(final String name, TrackLinkInterface startLink,
            TrackLinkInterface endLink, double radius) throws Exception
    {
        super(name, startLink, endLink);
        this.theta = Math.atan2(super.getdy(), super.getdx());
        this.radius = Math.abs(radius);
        Point2D start = new Point2D.Double(startLink.getPosition().x, startLink
                .getPosition().y);
        Point2D end = new Point2D.Double(endLink.getPosition().x, endLink
                .getPosition().y);
        Point2D[] intersection = Circle.intersection(start, this.radius, end,
                this.radius);
        if (intersection.length < 2)
            throw new Exception("ArcTrack " + name
                    + " - points too far apart for radius");
        this.counterClockwise = (radius < 0);
        if (this.counterClockwise)
            this.center = intersection[1];
        else
            this.center = intersection[0];
        this.startAngle = Angle.normalize2Pi(Angle.angle(this.center, start));
        this.endAngle = Angle.normalize2Pi(Angle.angle(this.center, end));
        if (this.counterClockwise)
        {
            this.deltaAngle = Angle.normalize2Pi(this.endAngle
                    - this.startAngle);
            this.length = this.radius
                    * Angle.normalize2Pi(this.endAngle - this.startAngle);
        } else
        {
            this.deltaAngle = -Angle.normalize2Pi(this.startAngle
                    - this.endAngle);
            this.length = this.radius
                    * Angle.normalize2Pi(this.startAngle - this.endAngle);
        }
        Logger.finest(this, "ArcTrack", "from:" + start.getX() + ","
                + start.getY() + "  to:" + end.getX() + "," + end.getY()
                + " - length=" + this.length + "  [as,ae=" + this.startAngle
                + "," + this.endAngle + "]" + "delta-a=" + this.deltaAngle);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getLocationOfProgression(double)
     */
    public DirectedPoint getLocationOfProgression(final double progression)
    {
        Point3d startPoint = this.startLink.getPosition();
        Point3d endPoint = this.endLink.getPosition();
        double delta = progression / this.length;
        double angle = Angle.normalize2Pi(this.startAngle + delta
                * this.deltaAngle);
        double x = this.center.getX() + this.radius * Math.cos(angle);
        double y = this.center.getY() + this.radius * Math.sin(angle);
        double z = startPoint.z + delta * (endPoint.z - startPoint.z);
        return new DirectedPoint(x, y, z, 0.0, 0.0, 0.0);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds()
    {
        Point3d p = this.startLink.getPosition();
        Point3d q = this.endLink.getPosition();
        return new BoundingBox(new Point3d(0, 0, 0), new Point3d(q.x - p.x, q.y
                - p.y, q.z - p.z));
    }

    /**
     * @return Returns the radius.
     */
    public double getRadius()
    {
        return this.radius;
    }

    /**
     * @return Returns the center.
     */
    public Point2D getCenter()
    {
        return this.center;
    }

    /**
     * @return Returns the endAngle.
     */
    public double getEndAngle()
    {
        return this.endAngle;
    }

    /**
     * @return Returns the startAngle.
     */
    public double getStartAngle()
    {
        return this.startAngle;
    }

    /**
     * @return Returns counter clockwise drawing direction.
     */
    public boolean isCounterClockwise()
    {
        return this.counterClockwise;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getLength()
     */
    public double getLength()
    {
        return this.length;
    }
}