/*
 * @(#)SimpleTrackLink.java Jun 19, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.track;

import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.track.util.TrackList;

/**
 * This is a complex track link, with several possible predecessors and several
 * possible successors. <br>
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version Jun 19, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class ComplexTrackLink extends DirectedPoint
        implements
            TrackLinkInterface
{
    /** the link name */
    private String name;

    /** the predecessor tracks */
    private TrackList predecessors = new TrackList();

    /** the successor tracks */
    private TrackList successors = new TrackList();

    /** the active predecessor for complex link */
    private TrackInterface activePredecessor;

    /** the active successor for complex link */
    private TrackInterface activeSuccessor;

    /**
     * Construct a new ComplexTrackLink
     * 
     * @param name
     * @param location
     *  
     */
    public ComplexTrackLink(final String name, final DirectedPoint location)
    {
        super(location);
        this.name = name;
    }

    /**
     * Construct a new ComplexTrackLink
     * 
     * @param name
     * @param point
     *  
     */
    public ComplexTrackLink(final String name, final Point3d point)
    {
        super(point);
        this.name = name;
    }

    /**
     * Construct a new ComplexTrackLink
     * 
     * @param name
     * @param x
     * @param y
     * @param z
     *  
     */
    public ComplexTrackLink(final String name, final double x, final double y,
            final double z)
    {
        super(x, y, z);
        this.name = name;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#addSuccessor(TrackInterface)
     */
    public void addSuccessor(final TrackInterface successor)
    {
        this.successors.add(successor);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#addPredecessor(TrackInterface)
     */
    public void addPredecessor(final TrackInterface predecessor)
    {
        this.predecessors.add(predecessor);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#setActiveSuccessor(TrackInterface,
     * TrackInterface)
     */
    public void setActiveSuccessor(final TrackInterface prevTrack,
            final TrackInterface nextTrack)
    {
        if (!this.predecessors.contains(prevTrack))
            Logger.warning(this, "setActiveSuccessor", "Predecessor "
                    + prevTrack + " for " + this.name
                    + " not in list of predecessor");
        if (!this.successors.contains(nextTrack))
            Logger.warning(this, "setActiveSuccessor", "Active successor "
                    + nextTrack + " for " + this.name
                    + " not in list of successors");
        this.activeSuccessor = nextTrack;
        setActivePredecessor(nextTrack, prevTrack);
        System.out.println("Sucessor changed successfully to:  "
                + nextTrack.toString());
    }

    /**
     * @param successor
     * @param predecessor
     */
    protected void setActivePredecessor(final TrackInterface successor,
            final TrackInterface predecessor)
    {
        if (!this.predecessors.contains(predecessor))
            Logger.warning(this, "setActivePredecessor", "Predecessor "
                    + predecessor + " for " + successor
                    + " not in list of predecessor");
        if (!this.successors.contains(successor))
            Logger.warning(this, "setActivePredecessor", "Active Successor "
                    + successor + " for " + predecessor
                    + " not in list of successors");
        this.activePredecessor = successor;
        System.out.println("Predecessor changed successfully to: "
                + predecessor.toString());
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getActiveSuccessor(TrackInterface)
     */
    public TrackInterface getActiveSuccessor(final TrackInterface predecessor)
    {
        if (!this.predecessors.contains(predecessor))
            Logger.warning(this, "getActiveSuccessor", "Predecessor "
                    + predecessor + " for " + this.name
                    + " not in list of predecessor");
        return this.activeSuccessor;
    }

    /**
     * @param successor
     * @return
     */
    public TrackInterface getActivePredecessor(final TrackInterface successor)
    {
        if (!this.successors.contains(successor))
            Logger.warning(this, "getActivePredecessor", "Successor "
                    + successor + " for " + this.name
                    + " not in list of predecessor");
        return this.activePredecessor;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getSuccessors()
     */
    public TrackList getSuccessors()
    {
        return this.successors;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getPredecessors()
     */
    public TrackList getPredecessors()
    {
        return this.predecessors;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getPosition()
     */
    public Point3d getPosition()
    {
        return this;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds()
    {
        return new BoundingSphere(this, 1.0);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation()
    {
        return this;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return this.name;
    }
}