/*
 * @(#)SimpleTrackLink.java Jun 19, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.track;

import java.rmi.RemoteException;

import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.vecmath.Point3d;

import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.track.util.TrackList;

/**
 * This is a simple track link, with one predecessor and one successor. <br>
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version Jun 19, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class SimpleTrackLink extends DirectedPoint
        implements
            TrackLinkInterface
{
    /** the link name */
    private String name;

    /** the predecessor track */
    private TrackInterface predecessor = null;

    /** the successor track */
    private TrackInterface successor = null;

    /**
     * Construct a new SimpleTrackLink
     * 
     * @param name
     * @param location
     *  
     */
    public SimpleTrackLink(final String name, final DirectedPoint location)
    {
        super(location);
        this.name = name;
    }

    /**
     * Construct a new SimpleTrackLink
     * 
     * @param name
     * @param point
     *  
     */
    public SimpleTrackLink(final String name, final Point3d point)
    {
        super(point);
        this.name = name;
    }

    /**
     * Construct a new SimpleTrackLink
     * 
     * @param name
     * @param x
     * @param y
     * @param z
     *  
     */
    public SimpleTrackLink(final String name, final double x, final double y,
            final double z)
    {
        super(x, y, z);
        this.name = name;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#addSuccessor(TrackInterface)
     */
    public void addSuccessor(final TrackInterface successor)
    {
        if (this.successor != null)
            Logger.warning(this, "addSuccessor", "Successor for " + this.name
                    + " already defined as " + this.successor + ", while "
                    + successor + " added");
        this.successor = successor;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#addPredecessor(TrackInterface)
     */
    public void addPredecessor(final TrackInterface predecessor)
    {
        if (this.predecessor != null)
            Logger.warning(this, "addPredecessor", "Predecessor for "
                    + this.name + " already defined as " + this.predecessor
                    + ", while " + predecessor + " added");
        this.predecessor = predecessor;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#setActiveSuccessor(TrackInterface,
     * TrackInterface)
     */
    public void setActiveSuccessor(final TrackInterface predecessor,
            final TrackInterface successor)
    {
        if (!predecessor.equals(this.predecessor))
            Logger.warning(this, "setActiveSuccessor", "Predecessor "
                    + predecessor + " for " + this.name
                    + " not in list of predecessor");
        if (!successor.equals(this.successor))
            Logger.warning(this, "setActiveSuccessor", "Active successor "
                    + successor + " for " + this.name
                    + " not in list of successors");
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getActiveSuccessor(TrackInterface)
     */
    public TrackInterface getActiveSuccessor(final TrackInterface predecessor)
    {
        if (!predecessor.equals(this.predecessor))
            Logger.warning(this, "getActiveSuccessor", "Predecessor "
                    + predecessor + " for " + this.name
                    + " not in list of predecessors");
        return this.successor;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getSuccessors()
     */
    public TrackList getSuccessors()
    {
        TrackList trackList = new TrackList();
        trackList.add(this.successor);
        return trackList;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getPredecessors()
     */
    public TrackList getPredecessors()
    {
        TrackList trackList = new TrackList();
        trackList.add(this.predecessor);
        return trackList;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface#getPosition()
     */
    public Point3d getPosition()
    {
        return this;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds()
    {
        return new BoundingSphere(this, 1.0);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        return this;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return this.name;
    }
}