/*
 * Created on Dec 4, 2003
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.track;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import nl.tudelft.simulation.dsol.animation.LocatableInterface;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface;
import nl.tudelft.simulation.traffic.controlpoint.util.ControlPointsList;
import nl.tudelft.simulation.traffic.track.util.TrackList;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * This class implements the TrackInterface. <br>
 * 
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a> <br>
 * 
 * Original authors: J.H. Kwakkel and H.W.G. Phaff
 */
public abstract class Track implements TrackInterface, LocatableInterface
{
    /** the name (id) of the track */
    protected String name;

    /** the start link */
    protected TrackLinkInterface startLink;

    /** the end link */
    protected TrackLinkInterface endLink;

    /** the control points on the track */
    private ControlPointsList controlPoints;

    /** the vehicles on the track */
    protected List vehicles = new ArrayList();

    /** all routes from a certain point */
    protected static final int TRACK_ROUTE_ALL = 0;

    /** active route from a certain point */
    protected static final int TRACK_ROUTE_ACTIVE = 1;

    /** line route from a certain point */
    protected static final int TRACK_ROUTE_LINE = 2;

    /**
     * @param name
     * @param startLink
     * @param endLink
     */
    public Track(final String name, final TrackLinkInterface startLink,
            final TrackLinkInterface endLink)
    {
        this.name = name;
        this.controlPoints = new ControlPointsList();
        this.startLink = startLink;
        this.endLink = endLink;
        startLink.addSuccessor(this);
        endLink.addPredecessor(this);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getStartLink()
     */
    public TrackLinkInterface getStartLink()
    {
        return this.startLink;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getEndLink()
     */
    public TrackLinkInterface getEndLink()
    {
        return this.endLink;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getControlPoints()
     */
    public ControlPointsList getControlPoints()
    {
        return this.controlPoints;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#addControlPoint(nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface)
     */
    public void addControlPoint(final ControlPointInterface cp)
    {
        this.controlPoints.add(cp);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getCpsOnInterval(
     * double, double)
     */
    public ControlPointsList getCpsOnInterval(final double progression,
            final double lengthOfInterval)
    {
        ControlPointsList list = new ControlPointsList();
        Iterator it = this.controlPoints.iterator();
        while (it.hasNext())
        {
            ControlPointInterface cp = (ControlPointInterface) it.next();
            double cpProg = cp.getProgression();
            if (cpProg >= progression
                    && cpProg <= progression + lengthOfInterval)
            {
                list.add(cp);
            }
        }
        return list;
    }

    /**
     * @return dx
     */
    public double getdx()
    {
        return (this.endLink.getPosition().x)
                - (this.startLink.getPosition().x);
    }

    /**
     * @return dy
     */
    public double getdy()
    {
        return (this.endLink.getPosition().y)
                - (this.startLink.getPosition().y);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation() throws RemoteException
    {
        return (DirectedPoint) this.startLink;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#removeControlPoint(nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface)
     */
    public void removeControlPoint(final ControlPointInterface cp)
    {
        this.controlPoints.remove(cp);
    }

    /**
     * @return Returns the name.
     */
    public String toString()
    {
        return this.name;
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#addVehicle(nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface)
     */
    public void addVehicle(VehiclePhysicalInterface vehicle)
    {
        if (this.vehicles.contains(vehicle))
            Logger.warning(this, "addVehicle", "vehicle already on track");
        this.vehicles.add(vehicle);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#removeVehicle(nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface)
     */
    public void removeVehicle(VehiclePhysicalInterface vehicle)
    {
        if (!this.vehicles.contains(vehicle))
            Logger.warning(this, "removeVehicle", "vehicle not on track "
                    + this);
        else if (this.vehicles.indexOf(vehicle) != 0)
            Logger.warning(this, "removeVehicle", "vehicle not first on track "
                    + this);
        else
            this.vehicles.remove(vehicle);
    }

    /**
     * @see nl.tudelft.simulation.traffic.track.TrackInterface#getVehiclesOnTrack()
     */
    public List getVehiclesOnTrack()
    {
        return this.vehicles;
    }

    /**
     * Calculate all positions on tracks located 'delta' meters from the start
     * of the current track. Delta can be positive or negative. When the track
     * splits, all paths are followed, and points on all the paths are returned.
     * 
     * @param delta
     * @return
     */
    public List calculateTrackProgressionListAll(final double delta)
    {
        List trackProgressionList = new ArrayList();
        if (delta == 0)
        {
            trackProgressionList.add(new TrackProgression(this, 0.0));
        } else if (delta > 0)
            calculateTrackProgressionListForward(this, delta,
                    Track.TRACK_ROUTE_ALL, trackProgressionList, null);
        else
        {
            calculateTrackProgressionListBackward(this, delta,
                    Track.TRACK_ROUTE_ALL, trackProgressionList, null);
        }
        if (trackProgressionList.size() == 0)
            System.out.println(this
                    + ", trackProgressionList.size() = 0. Track = " + this
                    + ", delta = " + delta);
        return trackProgressionList;
    }

    /**
     * Calculate a position on the 'active' track located 'delta' meters from
     * the start of the current track. When the track splits, only the active
     * path is followed, and a point on that path is returned. The active path
     * is determined by the current setting of the switches in the links that
     * connect the tracks.
     * 
     * @param delta
     * @return
     */
    public TrackProgression calculateTrackProgressionListActive(
            final double delta)
    {
        List trackProgressionList = new ArrayList();
        if (delta == 0)
        {
            trackProgressionList.add(new TrackProgression(this, 0.0));
        } else if (delta > 0)
            calculateTrackProgressionListForward(this, delta,
                    Track.TRACK_ROUTE_ACTIVE, trackProgressionList, null);
        else
            calculateTrackProgressionListBackward(this, delta,
                    Track.TRACK_ROUTE_ACTIVE, trackProgressionList, null);
        if (trackProgressionList.size() != 1)
        {
            System.out.println("ERROR, calculateTrackProgressionListActive. "
                    + "trackProgressionList.size() != 1 ("
                    + trackProgressionList.size() + "). Track = " + this
                    + ", delta = " + delta);
            if (trackProgressionList.size() > 1)
                return (TrackProgression) trackProgressionList.get(0);
            else
                return null;
        } else
            return (TrackProgression) trackProgressionList.get(0);
    }

    /**
     * Calculate a position on a track located 'delta' meters from the start of
     * the current track. Delta can (for now) only be positive. When the track
     * splits, the path that is linked to the given line number is followed, and
     * a point on that path is returned. This path is determined by the way the
     * switches will be set when a vehicle of the given line number passes.
     * 
     * @param delta
     * @param line
     * @return
     */
    public List calculateTrackProgressionListLine(final double delta,
            final String line)
    {
        List trackProgressionList = new ArrayList();
        if (delta == 0)
        {
            trackProgressionList.add(new TrackProgression(this, 0.0));
        } else if (delta > 0)
            calculateTrackProgressionListForward(this, delta,
                    Track.TRACK_ROUTE_LINE, trackProgressionList, line);
        else
            Logger.warning(this, "calculateTrackProgressionListLine",
                    "Negative delta cannot be calculated for line path");
        return trackProgressionList;
    }

    /**
     * This method gets a track and calculates the point(s) delta meters forward
     * on the basis of the START of the track.
     * 
     * @param track
     * @param delta
     * @param routeType
     * @param trackProgressionList
     * @param line
     */
    private void calculateTrackProgressionListForward(
            final TrackInterface track, final double delta,
            final int routeType, final List trackProgressionList,
            final String line)
    {
        double length = track.getLength();
        if (delta < length)
        {
            TrackProgression tp = new TrackProgression(track, delta);
            trackProgressionList.add(tp);
            return;
        }
        TrackLinkInterface link = track.getEndLink();
        switch (routeType)
        {
        case Track.TRACK_ROUTE_ALL:
            TrackList tl = link.getSuccessors();
            for (int i = 0; i < tl.size(); i++)
            {
                TrackInterface nextTrack = tl.get(i);
                if (nextTrack == null)
                {
                    System.err.println("TRACK " + track + ", LINK " + link
                            + " does not have successors");
                    return;
                }
                calculateTrackProgressionListForward(nextTrack, delta - length,
                        routeType, trackProgressionList, line);
            }
            break;
        case Track.TRACK_ROUTE_ACTIVE:
            TrackInterface nextTrack = link.getActiveSuccessor(track);
            if (nextTrack == null)
            {
                System.err.println("TRACK " + track + ", LINK " + link
                        + " does not have successors");
                return;
            }
            calculateTrackProgressionListForward(nextTrack, delta - length,
                    routeType, trackProgressionList, line);
            break;
        case Track.TRACK_ROUTE_LINE:
            Logger.warning(this, "calculateTrackProgressionListForward",
                    "Progression per line cannot be calculated yet");
            break;
        default:
            Logger.warning(this, "calculateTrackProgressionListForward",
                    "parameter 'all' does not have the right value: "
                            + routeType);
            break;
        }
    }

    /**
     * This method gets a track and has to calculate delta meters BACK on the
     * basis of the START of the track. When going backward, delta is a POSITIVE
     * number. When delta is NEGATIVE, the end point is probably ON this track.
     * 
     * @param track
     * @param delta
     * @param routeType
     * @param trackProgressionList
     * @param line
     */
    private void calculateTrackProgressionListBackward(
            final TrackInterface track, final double delta,
            final int routeType, final List trackProgressionList,
            final String line)
    {
        double length = track.getLength();
        if (delta > length)
        {
            // error, should be on this track
            Logger.warning(this, "calculateTrackProgressionListBackward",
                    "delta larger than track length");
            System.out.println("calculateTrackProgressionListBackward: "
                    + "delta " + delta + " larger than track " + track
                    + ".length " + length);
            return;
        }
        if (delta >= 0)
        {
            // on this track
            TrackProgression tp = new TrackProgression(track, delta);
            trackProgressionList.add(tp);
            return;
        }
        // on the previous track
        TrackLinkInterface link = track.getStartLink();
        if (routeType == Track.TRACK_ROUTE_ALL)
        {
            TrackList tl = link.getPredecessors();
            for (int i = 0; i < tl.size(); i++)
            {
                TrackInterface prevTrack = tl.get(i);
                calculateTrackProgressionListBackward(prevTrack, delta
                        + prevTrack.getLength(), routeType,
                        trackProgressionList, line);
            }
        } else if (routeType == Track.TRACK_ROUTE_ACTIVE)
        {
            TrackList tl = link.getPredecessors();
            if (tl.size() == 1)
            {
                TrackInterface prevTrack = tl.get(0);
                if (prevTrack == null)
                {
                    Logger.warning(this,
                            "calculateTrackProgressionListBackward",
                            "prevTrack = null");
                    System.out
                            .println("calculateTrackProgressionListBackward: "
                                    + "prevTrack = null for Track " + track);
                } else
                {
                    calculateTrackProgressionListBackward(prevTrack, delta
                            + prevTrack.getLength(), routeType,
                            trackProgressionList, line);
                }
            } else if (tl.size() > 1)
            {
                for (int i = 0; i < tl.size(); i++)
                {
                    TrackInterface prevTrack = tl.get(i);
                    if (track.equals(prevTrack.getEndLink().getActiveSuccessor(
                            prevTrack)))
                    {
                        calculateTrackProgressionListBackward(prevTrack, delta
                                + prevTrack.getLength(), routeType,
                                trackProgressionList, line);
                    }
                }
            } else
            {
                System.out
                        .println("ERROR calculateTrackProgressionListBackward: "
                                + "No predecessors for link " + link);
                return;
            }
        } else
        {
            Logger.warning(this, "calculateTrackProgressionListBackward",
                    "parameter 'all' does not have the right value: "
                            + routeType);
            return;
        }
    }

    /**
     * @param vehicle0
     * @param vehicle1
     * @return
     */
    public static double calculateDistanceFrontBack(
            final VehiclePhysicalInterface vehicle0,
            final VehiclePhysicalInterface vehicle1)
    {
        System.out
                .println("------------------------------------- calculateDistanceFrontBack "
                        + vehicle0 + "-" + vehicle1);
        // the maximum distance to check is 250 m
        double maxDistance = 250.0;
        // calculate the front location of vehicle0
        TrackProgression tp0 = new TrackProgression(vehicle0.getCurrentTrack(),
                vehicle0.getProgression());
        System.out.println("tp0 = " + tp0.getTrack() + ":"
                + tp0.getProgression());
        // calculate the back location of vehicle1
        TrackProgression tp1 = vehicle1.getCurrentTrack()
                .calculateTrackProgressionListActive(
                        vehicle1.getProgression()
                                - vehicle1.getVehicleType().getLength());
        System.out.println("tp1 = " + tp1.getTrack() + ":"
                + tp1.getProgression());
        // find tp1 forward from tp0. If it is not found, -1 will be returned
        double distance = calculateDistanceFrontBack(tp0, tp1, maxDistance);
        System.out.println("Distance between " + vehicle0 + " and " + vehicle1
                + " = " + distance);
        return distance;
    }

    /**
     * @param tp0
     * @param tp1
     * @param maxDistance
     * @return
     */
    public static double calculateDistanceFrontBack(final TrackProgression tp0,
            final TrackProgression tp1, final double maxDistance)
    {
        TrackInterface track = tp0.getTrack();
        double length = track.getLength() - tp0.getProgression();
        double distance = 0;
        while (distance < maxDistance)
        {
            if (track.equals(tp1.getTrack()))
            {
                return distance + tp1.getProgression();
            }
            distance += length;
            track = track.getEndLink().getActiveSuccessor(track);
            length = track.getLength();
        }
        return -1;
    }
}