/*
 * Created on Dec 4, 2003
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.track;

import java.util.List;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface;
import nl.tudelft.simulation.traffic.controlpoint.util.ControlPointsList;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;
import nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface;

/**
 * The TrackInterface defines the interface of Track. A track represents an
 * element of the infrastructure running from its startLink to its endLink.
 * 
 * Progression is defined as a distance along the infrastructure. Vehicles only
 * use progression. They only think in a line along the infrastructure, not in
 * worldcoordinates. <br>
 * 
 * @see nl.tudelft.simulation.traffic.track.TrackLinkInterface
 * 
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a> <br>
 * Original authors: J.H. Kwakkel and H.W.G. Phaff
 */
public interface TrackInterface
{
    /**
     * The getLocationOfProgression method is only used for animation. It
     * translates a progression on this track to a 3d-coordinate. It also
     * calculates the orientation of the returned Location.
     * 
     * @param progression
     * @return location
     */
    public DirectedPoint getLocationOfProgression(final double progression);

    /**
     * Returns controlpoints for the track, as a sorted list.
     * 
     * @return controlPoints
     */
    public ControlPointsList getControlPoints();

    /**
     * This method adds a controlPoint to this track.
     * 
     * @param cp
     * @see nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface
     */
    public void addControlPoint(final ControlPointInterface cp);

    /**
     * This method returns the link seen as the link from which the track
     * starts.
     * 
     * @return startLink
     */
    public TrackLinkInterface getStartLink();

    /**
     * This method returns the link seen as the link at which the track ends.
     * 
     * @return endLink
     */
    public TrackLinkInterface getEndLink();

    /**
     * @return length of the track
     */
    public double getLength();

    /**
     * This method checks for controlPoints in an interval, inclusive, so
     * returns cps that are on the edges of the interval
     * 
     * @param startProgression
     * @param lengthOfInterval
     * @return controlPoints
     */
    public ControlPointsList getCpsOnInterval(final double startProgression,
            final double lengthOfInterval);

    /**
     * This method removes a given cp from the ControlPointsLists of the track.
     * 
     * @param cp
     */
    public void removeControlPoint(final ControlPointInterface cp);

    /**
     * indicate that a vehicle is driving on the track (with front)
     * 
     * @param vehicle
     */
    public void addVehicle(VehiclePhysicalInterface vehicle);

    /**
     * indicate that a vehicle is no longer driving on the track (with front)
     * 
     * @param vehicle
     */
    public void removeVehicle(VehiclePhysicalInterface vehicle);

    /**
     * get the vehicles on the track
     * 
     * @return the list of vehicles
     */
    public List getVehiclesOnTrack();
    
    /**
     * Calculate all positions on tracks located 'delta' meters from the start
     * of the current track. Delta can be positive or negative. When the track
     * splits, all paths are followed, and points on all the paths are returned.
     * 
     * @param delta
     * @return
     */
    public List calculateTrackProgressionListAll(final double delta);

    /**
     * Calculate a position on the 'active' track located 'delta' meters from
     * the start of the current track. Delta can (for now) only be positive.
     * When the track splits, only the active path is followed, and a point on
     * that path is returned. The active path is determined by the current
     * setting of the switches in the links that connect the tracks.
     * 
     * @param delta
     * @return
     */
    public TrackProgression calculateTrackProgressionListActive(
            final double delta);

    /**
     * Calculate a position on a track located 'delta' meters from the start of
     * the current track. Delta can (for now) only be positive. When the track
     * splits, the path that is linked to the given line number is followed, and
     * a point on that path is returned. This path is determined by the way the
     * switches will be set when a vehicle of the given line number passes.
     * 
     * @param delta
     * @param line
     * @return
     */
    public List calculateTrackProgressionListLine(final double delta,
            final String line);
}