/*
 * @(#)DecelerationProfile.java Jul 5, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.vehicle;

import java.util.ArrayList;

import nl.tudelft.simulation.jstats.ode.DifferentialEquation;
import nl.tudelft.simulation.jstats.ode.integrators.NumericalIntegrator;

/**
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
 * Public License (GPL) </a>, no warranty <br>
 * 
 * @version Jul 5, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class AccelerationProfile
{
    /** the speed dependent acceleration / deceleration parameters */
    private ArrayList accdecList = new ArrayList();

    /** the maximum speed that we have in the arraylist */
    private double maxSpeed = 0.0;

    /** the dirty bit */
    private boolean initialized;

    /** the profile */
    private Profile profile;
    
    /** print table for debugging */
    protected static boolean debug = false;

    /**
     * create a new acceleration / deceleration profile
     * 
     * @param maxSpeed
     */
    public AccelerationProfile(final double maxSpeed)
    {
        super();
        this.initialized = false;
        this.maxSpeed = maxSpeed;
    }

    /**
     * Set the acceleration / deceleration for current speed HIGHER than speed.
     * Add the points in increasing speed. Sorting is not (yet) done
     * automatically.
     * 
     * @param speed the speed above which accdec is used
     * @param accdec the acceleration (positive) or deceleration (negative)
     */
    public void setAccelerationFromSpeed(final double speed, final double accdec)
    {
        this.accdecList.add(new AccDecPoint(speed, accdec));
        this.initialized = false;
    }

    /**
     * Gets the acceleration / deceleration for a given speed
     * 
     * @param speed
     * @return
     */
    public double getAcceleration(final double speed)
    {
        double accdec = 0.0;
        for (int i = 0; i < this.accdecList.size(); i++)
        {
            double adspeed = ((AccDecPoint) this.accdecList.get(i)).getSpeed();
            if (speed >= adspeed)
            {
                accdec = ((AccDecPoint) this.accdecList.get(i)).getAccDec();
            } else
            {
                return accdec;
            }
        }
        return accdec;
    }

    /**
     *  
     */
    public void initializeProfile()
    {
        this.profile = new Profile(this, this.maxSpeed);
        this.initialized = true;
    }

    /**
     * @param initialSpeed
     * @param endSpeed
     * @return the time to brake from the initialSpeed to the endSpeed
     */
    public double getAccelerationTime(final double initialSpeed,
            final double endSpeed)
    {
        if (!this.initialized)
            initializeProfile();
        return this.profile.getAccelerationTime(initialSpeed, endSpeed);
    }

    /**
     * @param initialSpeed
     * @param endSpeed
     * @return the distance to brake from the initialSpeed to the endSpeed
     */
    public double getAccelerationDistance(final double initialSpeed,
            final double endSpeed)
    {
        if (!this.initialized)
            initializeProfile();
        return this.profile.getAccelerationDistance(initialSpeed, endSpeed);
    }

    /**
     * Inner class to store the acc/dec points at certain speeds <br>
     * (c) <br>
     * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
     * University of Technology </a>, the Netherlands. <br>
     * See for project information <a href="http://www.simulation.tudelft.nl">
     * www.simulation.tudelft.nl </a> <br>
     * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
     * Public License (GPL) </a>, no warranty <br>
     * 
     * @version Jul 5, 2004 <br>
     * @author <a
     * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
     * Verbraeck </a>
     */
    private class AccDecPoint
    {
        /** the speed from which the acceleration/deceleration holds */
        private double speed;

        /** the acceleration/deceleration */
        private double accdec;

        /**
         * @param speed
         * @param accdec
         */
        public AccDecPoint(final double speed, final double accdec)
        {
            this.speed = speed;
            this.accdec = accdec;
        }

        /**
         * @return Returns the acceleration/deceleration.
         */
        public double getAccDec()
        {
            return this.accdec;
        }

        /**
         * @return Returns the speed.
         */
        public double getSpeed()
        {
            return this.speed;
        }
    }
    /**
     * Inner class to store the acceleration or deceleration profile based on
     * the speed and the acceleration / decelation panel. The class uses a
     * differential equation to make the calculations. <br>
     * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
     * University of Technology </a>, the Netherlands. <br>
     * See for project information <a href="http://www.simulation.tudelft.nl">
     * www.simulation.tudelft.nl </a> <br>
     * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
     * Public License (GPL) </a>, no warranty <br>
     * 
     * @version Jul 6, 2004 <br>
     * @author <a
     * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
     * Verbraeck </a>
     */
    private class Profile
    {
        /** the acceleration / deceleration profile */
        private AccelerationProfile owner;

        /** the list of speed to brake time */
        private double[] accTime;

        /** the list of speed to brake distance */
        private double[] accDistance;

        /** the resolution of the points */
        private double resolution = 0.1;

        /** the maximum speed that we have in the arraylist */
        private double maxSpeed = 0.0;

        /** the number of points in our profile */
        private int points;

        /**
         * @param owner
         * @param maxSpeed
         */
        public Profile(final AccelerationProfile owner, final double maxSpeed)
        {
            this.maxSpeed = maxSpeed;
            this.points = (int) (this.maxSpeed / this.resolution) + 1;
            this.owner = owner;
            this.accTime = new double[this.points];
            this.accDistance = new double[this.points];
            // instantiate the driving differential equation, and
            // calculate the speed and driven distance.
            Driving driving = new Driving(this.owner, 0.001,
                    NumericalIntegrator.RUNGEKUTTA4);
            double speed = 0.0;
            double x = 0.0;
            double time = 0.0;
            while (speed <= this.maxSpeed)
            {
                driving.initialize(time, new double[]{speed, x});
                time = time + 0.001;
                double[] vx = driving.y(time);
                speed = vx[0];
                double dx = vx[1] - x;
                x = vx[1];
                int i = getPoint(speed);
                this.accTime[i] += 0.001;
                this.accDistance[i] += dx;
            }
            for (int i = 0; i < this.points - 1; i++)
            {
                this.accTime[i + 1] += this.accTime[i];
                this.accDistance[i + 1] += this.accDistance[i];
            }
            if (AccelerationProfile.debug)
            {
            for (int i = 0; i < this.points; i++)
            {
                System.out.println("v=" + getSpeed(i) + ", [t,x]=["
                        + this.accTime[i] + "," + this.accDistance[i] + "]");
            }
            }
        }

        /**
         * @param speed
         * @return
         */
        private int getPoint(final double speed)
        {
            int point = (int) ((speed / this.maxSpeed) * this.points);
            return Math.min(this.points - 1, Math.max(point, 0));
        }

        /**
         * @param point
         * @return
         */
        private double getSpeed(final int point)
        {
            return this.resolution * point;
        }

        /**
         * @param initialSpeed
         * @param endSpeed
         * @return the time to accelerate from the initialSpeed to the endSpeed
         */
        public double getAccelerationTime(final double initialSpeed,
                final double endSpeed)
        {
            return this.accTime[getPoint(endSpeed)]
                    - this.accTime[getPoint(initialSpeed)];
        }

        /**
         * @param initialSpeed
         * @param endSpeed
         * @return the distance to accelerate from the initialSpeed to the
         * endSpeed
         */
        public double getAccelerationDistance(final double initialSpeed,
                final double endSpeed)
        {
            return this.accDistance[getPoint(endSpeed)]
                    - this.accDistance[getPoint(initialSpeed)];
        }

        /**
         * The differential equation for braking <br>
         * (c) copyright 2003-2004 <a
         * href="http://www.simulation.tudelft.nl">Delft University of
         * Technology </a>, the Netherlands. <br>
         * See for project information <a
         * href="http://www.simulation.tudelft.nl"> www.simulation.tudelft.nl
         * </a> <br>
         * License of use: <a
         * href="http://www.gnu.org/copyleft/gpl.html">General Public License
         * (GPL) </a>, no warranty <br>
         * 
         * @version Jul 6, 2004 <br>
         * @author <a
         * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
         * Verbraeck </a>
         */
        private class Driving extends DifferentialEquation
        {
            /** the acceleration / deceleration profile */
            private AccelerationProfile owner;

            /**
             * @param owner
             * @param timeStep
             */
            public Driving(final AccelerationProfile owner,
                    final double timeStep)
            {
                super(timeStep);
                this.owner = owner;
            }

            /**
             * @param owner
             * @param timeStep
             * @param integrator
             */
            public Driving(final AccelerationProfile owner,
                    final double timeStep, final NumericalIntegrator integrator)
            {
                super(timeStep, integrator);
                this.owner = owner;
            }

            /**
             * @param owner
             * @param timeStep
             * @param integrationMethod
             */
            public Driving(final AccelerationProfile owner,
                    final double timeStep, final short integrationMethod)
            {
                super(timeStep, integrationMethod);
                this.owner = owner;
            }

            /**
             * @see nl.tudelft.simulation.jstats.ode.DifferentialEquationInterface#dy(double,
             * double[])
             */
            public double[] dy(double x, double[] y)
            {
                // y[0] = v, y[1] = x
                double[] dy = new double[2];
                dy[0] = this.owner.getAcceleration(y[0]);
                dy[1] = y[0];
                return dy;
            }
        }
    }

    /**
     * @param args
     */
    public static void main(String[] args)
    {
        try
        {
            AccelerationProfile dp = new AccelerationProfile(14.0); // m/s
            dp.setAccelerationFromSpeed(0.0, 1.0);
            dp.initializeProfile();
            System.out.println("Acceleration distance from 0.0 to 14.0 = "
                    + dp.getAccelerationDistance(0.0, 14.0));
            System.out.println("Acceleration distance from 7.0 to 14.0 = "
                    + dp.getAccelerationDistance(7.0, 14.0));
            System.out.println("Acceleration distance from  0.0 to 7.0 = "
                    + dp.getAccelerationDistance(0.0, 7.0));
            System.out.println("Acceleration time from 0.0 to 14.0 = "
                    + dp.getAccelerationTime(0.0, 14.0));
            System.out.println("Acceleration time from 7.0 to 14.0 = "
                    + dp.getAccelerationTime(7.0, 14.0));
            System.out.println("Acceleration time from  0.0 to 7.0 = "
                    + dp.getAccelerationTime(0.0, 7.0));
        } catch (Exception e)
        {
            e.printStackTrace();
        }
    }
}
