/*
 * @(#)VehiclePhysical.java Jul 4, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.vehicle;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.media.j3d.Bounds;
import javax.media.j3d.Transform3D;
import javax.vecmath.Point3d;
import nl.tudelft.simulation.dsol.simulators.AnimatorInterface;
import nl.tudelft.simulation.dsol.simulators.DEVDESSSimulatorInterface;
import nl.tudelft.simulation.dsol.statistics.charts.XYChart;
import nl.tudelft.simulation.event.EventInterface;
import nl.tudelft.simulation.event.EventProducer;
import nl.tudelft.simulation.event.EventProducerInterface;
import nl.tudelft.simulation.event.EventType;
import nl.tudelft.simulation.event.TimedEvent;
import nl.tudelft.simulation.language.d3.BoundingBox;
import nl.tudelft.simulation.language.d3.DirectedPoint;
import nl.tudelft.simulation.logger.Logger;
import nl.tudelft.simulation.traffic.animation.VehicleAnimation;
import nl.tudelft.simulation.traffic.controlpoint.ControlPointInterface;
import nl.tudelft.simulation.traffic.controlpoint.real.Changeable;
import nl.tudelft.simulation.traffic.controlpoint.real.SpeedLimitInterface;
import nl.tudelft.simulation.traffic.controlpoint.real.StopSignInterface;
import nl.tudelft.simulation.traffic.controlpoint.util.ControlPointsList;
import nl.tudelft.simulation.traffic.track.Track;
import nl.tudelft.simulation.traffic.track.TrackInterface;
import nl.tudelft.simulation.traffic.track.TrackLinkInterface;
import nl.tudelft.simulation.traffic.track.util.TrackProgression;

/**
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General Public
 * License (GPL) </a>, no warranty <br>
 * 
 * @version Jul 4, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class VehiclePhysical extends EventProducer
        implements
            VehiclePhysicalInterface
{
    /** the simulator to schedule on */
    private DEVDESSSimulatorInterface simulator;

    /** the vehicle type with all relevant information */
    private VehicleType vehicleType;

    /** vehicle control of this vehicle */
    private VehicleControlInterface vehicleControl;

    /** current track to travel on */
    private List currentTracks = new ArrayList();

    /** current progression on the current track */
    private double progression;

    /** current speed */
    private double currentSpeed = 0.0;

    /** total distance traveled since creation */
    private double distanceTraveled;

    /** are we allowed to drive? */
    private boolean driving = false;

    /** the differential equations to make the vehicle move */
    private VehiclePositioner vehiclePositioner;

    /** current maximum speed */
    private double currentMaximumSpeed;

    /** acceleration profile */
    private AccelerationProfile accelerationProfile;

    /** deceleration profile */
    private DecelerationProfile decelerationProfile;

    /** the progression horizon to look for control points */
    private double horizon;

    /** the control points between the current position and the horizon */
    private List speedControlPoints = new ArrayList();

    /** the array of control points on the current track */
    private List trackControlPointsList = new ArrayList();

    /** the unique name */
    private String name;

    /** the chart to display the speed vs time graph */
    private XYChart tvChart;

    /** the chart to display the speed vs progression graph */
    private XYChart xvChart;

    /** the t-v event type to use for the xyChart */
    public EventType TIME_SPEED_1SECOND_EVENT = new EventType("tv");

    /** the x-v event type to use for the xyChart */
    public EventType DISTANCE_SPEED_1SECOND_EVENT = new EventType("xv");

    /** last second in the chart */
    private double sampleSecond = -10.0;

    /** helper string */
    private String lastString;

    /** the maximum braking distance */
    private double maxBrakeDistance;

    /**
     * Create a physical vehicle
     * 
     * @param vehicleType
     * @param vehicleControl
     * @param track
     * @param progression
     * @param maxSpeedStart
     * @param simulator
     */
    public VehiclePhysical(final VehicleType vehicleType,
            final VehicleControlInterface vehicleControl,
            final TrackInterface track, final double progression,
            final double maxSpeedStart,
            final DEVDESSSimulatorInterface simulator)
    {
        try
        {
            this.simulator = simulator;
            this.distanceTraveled = 0;
            this.name = vehicleType.generateUniqueName();
            this.vehicleType = vehicleType;
            this.accelerationProfile = vehicleType.getAccelerationProfile();
            this.decelerationProfile = vehicleType.getDecelerationProfile();
            this.maxBrakeDistance = this.decelerationProfile.getBrakeDistance(
                    vehicleType.getVehicleMaximumSpeed(), 0.0);
            this.horizon = this.maxBrakeDistance * 1.5;
            System.out.println("Horizon for vehicle " + this.name + " is "
                    + this.horizon);
            this.vehicleControl = vehicleControl;
            moveToFirstTrack(track, progression);
            this.currentMaximumSpeed = maxSpeedStart;
            if (simulator instanceof AnimatorInterface)
            {
                ArrayList segments = this.vehicleType.getSegments();
                if (segments.isEmpty())
                {
                    new VehicleAnimation(this, simulator, this.vehicleType
                            .getColor());
                } else
                {
                    for (int i = 0; i < segments.size(); i++)
                    {
                        VehicleType.Segment segment = (VehicleType.Segment) segments
                                .get(i);
                        new VehicleSegment(this, simulator, vehicleType
                                .getColor(), segment.getFront(), segment
                                .getAxle1(), segment.getAxle2(), segment
                                .getLength());
                    }
                }
                // make a t/v graph for this vehicle
                this.tvChart = new XYChart(simulator, "t-v vehicle "
                        + this.name);
                this.tvChart.getChart().getXYPlot().getDomainAxis()
                        .setAutoRange(true);
                this.tvChart.getChart().getXYPlot().getDomainAxis().setLabel(
                        "time (s)");
                this.tvChart.getChart().getXYPlot().getRangeAxis()
                        .setAutoRange(false);
                this.tvChart.getChart().getXYPlot().getRangeAxis()
                        .setLowerBound(0.0);
                this.tvChart.getChart().getXYPlot().getRangeAxis()
                        .setUpperBound(
                                this.vehicleType.getVehicleMaximumSpeed());
                this.tvChart.getChart().getXYPlot().getRangeAxis().setLabel(
                        "speed (m/s)");
                this.TIME_SPEED_1SECOND_EVENT = new EventType("t-v vehicle "
                        + this.name);
                this.tvChart.add("t-v vehicle " + this.name, this,
                        this.TIME_SPEED_1SECOND_EVENT);
                // make an x/v graph for this vehicle
                this.xvChart = new XYChart(simulator, "x-v vehicle "
                        + this.name);
                this.xvChart.getChart().getXYPlot().getDomainAxis()
                        .setAutoRange(true);
                this.xvChart.getChart().getXYPlot().getDomainAxis().setLabel(
                        "progression (m)");
                this.xvChart.getChart().getXYPlot().getRangeAxis()
                        .setAutoRange(false);
                this.xvChart.getChart().getXYPlot().getRangeAxis()
                        .setLowerBound(0.0);
                this.xvChart.getChart().getXYPlot().getRangeAxis()
                        .setUpperBound(
                                this.vehicleType.getVehicleMaximumSpeed());
                this.xvChart.getChart().getXYPlot().getRangeAxis().setLabel(
                        "speed (m/s)");
                this.DISTANCE_SPEED_1SECOND_EVENT = new EventType(
                        "x-v vehicle " + this.name);
                this.xvChart.add("x-v vehicle " + this.name, this,
                        this.DISTANCE_SPEED_1SECOND_EVENT);
            }
            this.vehiclePositioner = new VehiclePositioner(this, simulator);
            this.vehiclePositioner.setValue(0.0, progression);
        } catch (Exception exception)
        {
            exception.printStackTrace();
        }
    }

    /**
     * @see nl.tudelft.simulation.event.EventProducerInterface#getEventTypes()
     */
    public synchronized EventType[] getEventTypes()
    {
        return new EventType[]{};
    }

    /**
     * rebuilds the list of control points between the current location and the
     * horizon. Always take at least two tracks, where the second track should
     * be longer than the max brake distance.
     */
    private void readControlPointsWithinHorizon()
    {
        Logger.fine(this, "readControlPointsWithinHorizon", this.name
                + ": Making new horizon list for track "
                + this.getCurrentTrack());
        this.speedControlPoints.clear();
        // loop through the tracks, and store all control points we encounter
        TrackInterface track = this.getCurrentTrack();
        double progression = 0.0;
        while (progression < this.horizon + this.getCurrentTrack().getLength())
        {
            if (track == null)
                return;
            // take whole tracks
            ControlPointsList cpList = track.getControlPoints();
            Iterator it = cpList.iterator();
            while (it.hasNext())
            {
                ControlPointInterface cp = (ControlPointInterface) it.next();
                // subscribe to the changable ones
                if (cp instanceof Changeable)
                {
                    try
                    {
                        ((EventProducerInterface) cp).addListener(this,
                                Changeable.CHANGE_STATUS_EVENT);
                    } catch (Exception e)
                    {
                        Logger
                                .severe(this, "readControlPointsWithinHorizon",
                                        e);
                    }
                }
                // look at the ones that (might) limit speed or are changable
                // and add these to the list, with the right progression
                // from the start of the first track in the list
                if ((cp instanceof Changeable)
                        || (cp instanceof SpeedLimitInterface)
                        || (cp instanceof StopSignInterface))
                {
                    ControlPointProgression cpp = new ControlPointProgression(
                            cp, progression + cp.getProgression());
                    this.speedControlPoints.add(cpp);
                    Logger.fine(this, "readControlPointsWithinHorizon",
                            "   ..." + cp + ", distance = " + progression
                                    + cp.getProgression());
                }
            }
            progression += track.getLength();
            TrackLinkInterface link = track.getEndLink();
            track = link.getActiveSuccessor(getCurrentTrack());
        }
    }

    /**
     * indicate that the vehicle can start driving. As long as this variable is
     * false, the acceleration remains zero.
     * 
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#startDriving()
     */
    public void startDriving()
    {
        this.driving = true;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getCurrentAcceleration(double,
     * double)
     */
    public double getCurrentAcceleration(final double speed,
            final double progression)
    {
        try
        {
            double time = this.simulator.getSimulatorTime();
            // sample max 1 times per second
            if (time - this.sampleSecond > 1.0)
            {
                this.fireEvent(new TimedEvent(this.TIME_SPEED_1SECOND_EVENT,
                        this, new Double(speed), time));
                this.fireEvent(new TimedEvent(
                        this.DISTANCE_SPEED_1SECOND_EVENT, this, new Double(
                                speed), this.distanceTraveled));
                this.sampleSecond = time;
            }
        } catch (RemoteException e)
        {
            e.printStackTrace();
        }
        // this is the method that determines, based on the restrictions,
        // whether to decelerate, accelerate, or drive constant. It should
        // be efficient, as it is called in each timestep for each vehicle.
        if (!this.driving)
        {
            return 0.0;
        }
        // look if we can still accelerate
        double accdec = 0.0;
        if (speed < this.currentMaximumSpeed)
        {
            accdec = this.accelerationProfile.getAcceleration(speed);
        }
        // look if we need to brake; look at the current brake distance and
        // see if there are any speed limits within the interval, that need
        // our attention
        // cheat a little bit: brake 1 centimeters before stop point.
        // add one meter for the moment, because the deceleration is one
        // meter off -- TODO: check the brake distance
        double brakeDistance0 = this.decelerationProfile.getBrakeDistance(
                speed, 0) + 1.01;
        // Look if we are too close to a predecessor. If so, brake.
        double predecessorDistance = calculatePredecessorDistance(progression,
                brakeDistance0);
        if (predecessorDistance < brakeDistance0)
        {
            if (speed > 0)
                return this.decelerationProfile.getDeceleration(speed);
            else
                return 0.0;
        }
        // See if there are speedpoints
        Iterator it = this.speedControlPoints.iterator();
        boolean finished = false;
        while (it.hasNext() && !finished)
        {
            ControlPointProgression cpp = (ControlPointProgression) it.next();
            ControlPointInterface cp = cpp.getControlPoint();
            double cpDistance = cpp.getDistance() - progression;
            if (cpDistance > brakeDistance0)
                finished = true;
            // if < 0, we already passed the point, but the list was
            // regenerated for the WHOLE track -- e.g. after a Changeable
            // event, or another disruption.
            if (cpDistance > 0.0)
            {
                // look if we need to brake for this point
                if (cp instanceof SpeedLimitInterface)
                {
                    if (cpDistance <= this.decelerationProfile
                            .getBrakeDistance(speed, ((SpeedLimitInterface) cp)
                                    .getSpeedLimit()))
                    {
                        if (!cp.toString().equals(this.lastString))
                        {
                            this.lastString = cp.toString();
                            Logger
                                    .fine(
                                            this,
                                            "getCurrentAcceleration",
                                            this.name
                                                    + " braking for speedlimit cp "
                                                    + this.lastString
                                                    + ", cpDistance="
                                                    + cpDistance
                                                    + "\n   brakeDistance="
                                                    + this.decelerationProfile
                                                            .getBrakeDistance(
                                                                    speed,
                                                                    ((SpeedLimitInterface) cp)
                                                                            .getSpeedLimit())
                                                    + ", progression="
                                                    + this.progression
                                                    + "\n   track="
                                                    + this.getCurrentTrack()
                                                    + ", length="
                                                    + this.getCurrentTrack()
                                                            .getLength());
                        }
                        accdec = this.decelerationProfile
                                .getDeceleration(speed);
                        finished = true;
                    }
                }
                // look if we need to stop for this point
                else if (cp instanceof StopSignInterface)
                {
                    if (((StopSignInterface) cp).getStatus().equals(
                            StopSignInterface.STOP)
                            && (cpDistance <= brakeDistance0))
                    {
                        if (!cp.toString().equals(this.lastString))
                        {
                            this.lastString = cp.toString();
                            Logger.fine(this, "getCurrentAcceleration",
                                    this.name
                                            + " braking for trafficlight cp "
                                            + this.lastString
                                            + ", cpDistance="
                                            + cpDistance
                                            + "\n   brakeDistance="
                                            + brakeDistance0
                                            + ", progression="
                                            + this.progression
                                            + "\n   track="
                                            + this.getCurrentTrack()
                                            + ", length="
                                            + this.getCurrentTrack()
                                                    .getLength());
                        }
                        accdec = this.decelerationProfile
                                .getDeceleration(speed);
                        finished = true;
                    }
                    // look if we are 'close' to the traffic light;
                    // if it is red, we also stop.
                    if (((StopSignInterface) cp).getStatus().equals(
                            StopSignInterface.STOP)
                            && (cpDistance <= 1.01))
                    {
                        if (speed > 0)
                            accdec = this.decelerationProfile
                                    .getDeceleration(speed);
                        else
                            accdec = 0.0;
                    }
                }
            }
        }
        return accdec;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#setSpeedAndProgression(double,
     * double)
     */
    public double[] setSpeedAndProgression(final double speed,
            final double progression)
    {
        this.currentSpeed = speed;
        // the progression is increased with the new progression minus
        // the previous progression
        this.distanceTraveled += progression - this.progression;
        TrackInterface track = this.getCurrentTrack();
        // check if we hit any control points. Work on the basis of just the
        // next control point, but check if there are multiple control points
        // on or near the same location
        if (progression < track.getLength())
        {
            driveOverControlPoints(this.progression, progression);
            this.progression = progression;
        } else
        {
            // we moved off the track; handle the control points of both this
            // track and the next.
            driveOverControlPoints(this.progression, track.getLength());
            double restProgression = Math.max(0.0, track.getLength()
                    - this.progression);
            moveToNextTrack();
            track = this.getCurrentTrack();
            driveOverControlPoints(0.0, restProgression);
            this.vehiclePositioner.setValue(this.currentSpeed, restProgression);
            this.progression = restProgression;
        }
        return new double[]{this.currentSpeed, this.progression};
    }

    /**
     * The front of the vehicle passes the track limit. Add a new track to the
     * current track list. The progression on the new track will be 0. The
     * 'update' method is responsible for deleting trailing tracks that are no
     * longer needed.
     * 
     * Note: the vehicle is only registered on the track of the front of the
     * vehicle.
     */
    protected void moveToNextTrack()
    {
        getCurrentTrack().removeVehicle(this);
        TrackLinkInterface link = getCurrentTrack().getEndLink();
        TrackInterface newCurrentTrack = link
                .getActiveSuccessor(getCurrentTrack());
        this.progression = 0;
        this.currentTracks.add(newCurrentTrack);
        Logger.fine(this, "moveToNextTrack", this.name
                + " moved to next track " + this.getCurrentTrack());
        readControlPointsWithinHorizon();
        makeControlPointsArray(newCurrentTrack);
        newCurrentTrack.addVehicle(this);
    }

    /**
     * @param track
     * @param progression
     */
    protected void moveToFirstTrack(final TrackInterface track,
            final double progression)
    {
        this.currentTracks.add(track);
        this.progression = progression;
        this.currentTracks.add(track);
        readControlPointsWithinHorizon();
        makeControlPointsArray(track);
        track.addVehicle(this);
    }

    /**
     * @param progression
     * @param maxDistance
     * @return
     */
    private double calculatePredecessorDistance(final double progression,
            final double maxDistance)
    {
        double rest = maxDistance;
        TrackInterface track = getCurrentTrack();
        double length = track.getLength() - progression;
        while (rest > 0)
        {
            List vehicleList = track.getVehiclesOnTrack();
            for (int i = 0; i < vehicleList.size(); i++)
            {
                VehiclePhysicalInterface vehicle = (VehiclePhysicalInterface) vehicleList
                        .get(i);
                if (vehicle != this)
                {
                    double distance = Track.calculateDistanceFrontBack(this,
                            vehicle);
                    if (distance > 0)
                        return distance;
                }
            }
            rest -= length;
            track = track.getEndLink().getActiveSuccessor(track);
            if (track == null)
                return (maxDistance + 1.0);
            length = track.getLength();
        }
        return (maxDistance + 1.0);
    }

    /**
     * @param track
     */
    private void makeControlPointsArray(final TrackInterface track)
    {
        this.trackControlPointsList.clear();
        ControlPointsList cpl = track.getControlPoints();
        Iterator it = cpl.iterator();
        while (it.hasNext())
        {
            ControlPointInterface cp = (ControlPointInterface) it.next();
            this.trackControlPointsList.add(cp);
        }
    }

    /**
     * @param from
     * @param to
     */
    protected void driveOverControlPoints(final double from, final double to)
    {
        boolean finished = false;
        while (!finished)
        {
            if (this.trackControlPointsList.size() == 0)
                finished = true;
            else
            {
                ControlPointInterface cp = (ControlPointInterface) this.trackControlPointsList
                        .get(0);
                if (cp.getProgression() >= from - 0.001
                        && cp.getProgression() <= to + 0.001)
                {
                    System.out.println(this.name + " drove over cp " + cp);
                    this.trackControlPointsList.remove(0);
                    // see if we can also remove it from the speedControlPoints
                    Iterator it = this.speedControlPoints.iterator();
                    while (it.hasNext())
                    {
                        ControlPointProgression cpp = (ControlPointProgression) it
                                .next();
                        if (cpp.getControlPoint().equals(cp))
                        {
                            System.out
                                    .println("also removed from speedControlPoints: "
                                            + cp);
                            it.remove();
                        }
                    }
                    cp.pass(this);
                } else
                {
                    finished = true;
                }
            }
        }
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getCurrentSpeed()
     */
    public double getCurrentSpeed()
    {
        return this.currentSpeed;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#setMaximumSpeed(double)
     */
    public void setMaximumSpeed(final double maxSpeed)
    {
        this.currentMaximumSpeed = maxSpeed;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getProgression()
     */
    public double getProgression()
    {
        return this.progression;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getCurrentTrack()
     */
    public TrackInterface getCurrentTrack()
    {
        return (TrackInterface) this.currentTracks.get(this.currentTracks
                .size() - 1);
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getVehicleType()
     */
    public VehicleType getVehicleType()
    {
        return this.vehicleType;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getVehicleControl()
     */
    public VehicleControlInterface getVehicleControl()
    {
        return this.vehicleControl;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#setVehicleControl(nl.tudelft.simulation.traffic.vehicle.VehicleControlInterface)
     */
    public void setVehicleControl(VehicleControlInterface vehicleControl)
    {
        this.vehicleControl = vehicleControl;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#getDistanceTraveled()
     */
    public double getDistanceTraveled()
    {
        return this.distanceTraveled;
    }

    /**
     * @see nl.tudelft.simulation.traffic.vehicle.VehiclePhysicalInterface#removeVehicle()
     */
    public void removeVehicle() throws RemoteException
    {
        // the vehicle is at this moment only registered on the front track
        getCurrentTrack().removeVehicle(this);
        this.vehicleControl.removeVehicle();
        this.fireEvent(VehiclePhysicalInterface.REMOVE_EVENT, true);
        removeAllListeners();
        this.vehiclePositioner.stopIntegration();
        this.driving = false;
    }

    /**
     * @see nl.tudelft.simulation.event.EventListenerInterface#notify(nl.tudelft.simulation.event.EventInterface)
     */
    public void notify(EventInterface event) throws RemoteException
    {
        if (event.getType().equals(Changeable.CHANGE_STATUS_EVENT))
        {
            if (!this.driving)
            {
                ((EventProducerInterface) event.getSource()).removeListener(
                        this, Changeable.CHANGE_STATUS_EVENT);
            }
            this.readControlPointsWithinHorizon();
        } else
            System.err.println("Vehicle " + this.name + ": unknown event: "
                    + event);
    }

    /**
     * @return the total length of the current track list
     */
    protected double getCurrentTracksLength()
    {
        double totalLength = 0.0;
        for (int i = 0; i < this.currentTracks.size(); i++)
        {
            totalLength += ((TrackInterface) this.currentTracks.get(i))
                    .getLength();
        }
        return totalLength;
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getLocation()
     */
    public DirectedPoint getLocation()
    {
        // for this moment: determine direction on basis of total length
        // measure along the path
        DirectedPoint front = getCurrentTrack().getLocationOfProgression(
                this.progression);
        TrackProgression tp = getCurrentTrack()
                .calculateTrackProgressionListActive(
                        this.progression - this.vehicleType.getLength());
        if (tp == null)
        {
            tp = getCurrentTrack().calculateTrackProgressionListActive(
                    this.progression - 0.1);
        }
        if (tp == null)
            return new DirectedPoint(front.x, front.y, 0.1, 0, 0, 0);
        DirectedPoint back = tp.getTrack().getLocationOfProgression(
                tp.getProgression());
        double theta = Math.atan2(back.y - front.y, back.x - front.x);
        return new DirectedPoint(front.x, front.y, front.z + 0.05, 0, 0, theta);
    }

    /**
     * get a location along the track to the back
     * 
     * @param distance
     * @return the location
     */
    public Point3d getBackwardLocation(final double distance)
    {
        TrackProgression tp = getCurrentTrack()
                .calculateTrackProgressionListActive(
                        this.progression - distance);
        if (tp == null)
        {
            System.err.println("ERROR " + this.name
                    + ": cannot calculate backwardLocation from "
                    + getCurrentTrack() + ", progression=" + this.progression
                    + ", back distance=" + distance);
            tp = getCurrentTrack().calculateTrackProgressionListActive(
                    this.progression);
            if (tp == null)
                return new DirectedPoint(0, 0, 0);
        }
        DirectedPoint back = tp.getTrack().getLocationOfProgression(
                tp.getProgression());
        return new Point3d(back.x, back.y, back.z);
    }

    /**
     * @see nl.tudelft.simulation.dsol.animation.LocatableInterface#getBounds()
     */
    public Bounds getBounds()
    {
        DirectedPoint location = getLocation();
        double width = this.vehicleType.getWidth();
        Bounds box = new BoundingBox(new Point3d(0.0, -width / 2.0, 0.0),
                new Point3d(this.vehicleType.getLength(), width / 2.0,
                        0.0));
        Transform3D rot = new Transform3D();
        rot.rotZ(location.getRotZ());
        box.transform(rot);
        return box;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return this.name;
    }

    /**
     * 
     * <br>
     * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
     * University of Technology </a>, the Netherlands. <br>
     * See for project information <a href="http://www.simulation.tudelft.nl">
     * www.simulation.tudelft.nl </a> <br>
     * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
     * Public License (GPL) </a>, no warranty <br>
     * 
     * @version Jul 7, 2004 <br>
     * @author <a
     * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
     * Verbraeck </a>
     */
    private class ControlPointProgression
    {
        /** the control point */
        private ControlPointInterface controlPoint;

        /** the overall progression from the start of the first track */
        private double distance;

        /**
         * @param controlPoint
         * @param distance
         */
        public ControlPointProgression(
                final ControlPointInterface controlPoint, final double distance)
        {
            this.controlPoint = controlPoint;
            this.distance = distance;
        }

        /**
         * @return Returns the controlPoint.
         */
        public ControlPointInterface getControlPoint()
        {
            return this.controlPoint;
        }

        /**
         * @return Returns the distance.
         */
        public double getDistance()
        {
            return this.distance;
        }

        /**
         * @see java.lang.Object#toString()
         */
        public String toString()
        {
            return "CPP:" + this.controlPoint + ", distance:" + this.distance;
        }
    }
}