/*
 * @(#)VehiclePositioner.java Jul 4, 2004
 * 
 * Copyright (c) 2003, 2004 Delft University of Technology Jaffalaan 5, 2628 BX
 * Delft, the Netherlands All rights reserved.
 * 
 * This software is proprietary information of Delft University of Technology
 * The code is published under the General Public License
 */

package nl.tudelft.simulation.traffic.vehicle;

import java.rmi.RemoteException;

import nl.tudelft.simulation.dsol.formalisms.dess.DifferentialEquation;
import nl.tudelft.simulation.dsol.simulators.DESSSimulatorInterface;
import nl.tudelft.simulation.dsol.simulators.SimulatorInterface;
import nl.tudelft.simulation.jstats.ode.integrators.NumericalIntegrator;
import nl.tudelft.simulation.logger.Logger;

/**
 * <br>
 * (c) copyright 2003-2004 <a href="http://www.simulation.tudelft.nl">Delft
 * University of Technology </a>, the Netherlands. <br>
 * See for project information <a href="http://www.simulation.tudelft.nl">
 * www.simulation.tudelft.nl </a> <br>
 * License of use: <a href="http://www.gnu.org/copyleft/gpl.html">General
 * Public License (GPL) </a>, no warranty <br>
 * 
 * @version Jul 4, 2004 <br>
 * @author <a
 * href="http://www.tbm.tudelft.nl/webstaf/alexandv/index.htm">Alexander
 * Verbraeck </a>
 */
public class VehiclePositioner extends DifferentialEquation
{
    /** the vehicle */
    protected VehiclePhysicalInterface vehiclePhysical;

    /**
     * @param vehiclePhysical
     * @param simulator
     * @throws RemoteException
     */
    public VehiclePositioner(final VehiclePhysicalInterface vehiclePhysical,
            final DESSSimulatorInterface simulator) throws RemoteException
    {
        this(vehiclePhysical, simulator, simulator.getTimeStep(),
                NumericalIntegrator.DEFAULT_INTEGRATOR);
    }

    /**
     * @param vehiclePhysical
     * @param simulator
     * @param timeStep
     */
    public VehiclePositioner(final VehiclePhysicalInterface vehiclePhysical,
            final DESSSimulatorInterface simulator, final double timeStep)
    {
        this(vehiclePhysical, simulator, timeStep,
                NumericalIntegrator.DEFAULT_INTEGRATOR);
    }

    /**
     * @param vehiclePhysical
     * @param simulator
     * @param timeStep
     * @param numericalMethod
     */
    public VehiclePositioner(final VehiclePhysicalInterface vehiclePhysical,
            final DESSSimulatorInterface simulator, final double timeStep,
            final short numericalMethod)
    {
        super(simulator, timeStep, numericalMethod);
        this.vehiclePhysical = vehiclePhysical;
        this.initialize(0.0, new double[]{0.0, 0.0});
    }

    /**
     * @param vehiclePhysical
     * @param simulator
     * @param timeStep
     * @param numericalIntegrator
     */
    public VehiclePositioner(final VehiclePhysical vehiclePhysical,
            final DESSSimulatorInterface simulator, final double timeStep,
            final NumericalIntegrator numericalIntegrator)
    {
        super(simulator, timeStep, numericalIntegrator);
        this.vehiclePhysical = vehiclePhysical;
        this.initialize(0.0, new double[]{0.0, 0.0});
    }

    /**
     * sets the speed and progression (e.g. when entering a new track, the
     * progression will be reset).
     * 
     * @param speed
     * @param progression
     *  
     */
    public void setValue(final double speed, final double progression)
    {
        try
        {
            super.initialize(this.simulator.getSimulatorTime(), new double[]{
                    speed, progression});
        } catch (RemoteException exception)
        {
            Logger.warning(this, "setValue", exception);
        }
    }

    /**
     * @see nl.tudelft.simulation.jstats.ode.DifferentialEquationInterface#dy(double,
     * double[])
     */
    public double[] dy(double x, double[] y)
    {
        // y[0] = v, y[1] = x
        double[] dy = new double[2];
        dy[0] = this.vehiclePhysical.getCurrentAcceleration(y[0], y[1]);
        dy[1] = y[0];
        return dy;
    }

    /**
     * @see nl.tudelft.simulation.jstats.ode.DifferentialEquation#integrateY(double,
     * double, double[])
     */
    protected double[] integrateY(double x, double initialX, double[] initialY)
    {
        double[] y = super.integrateY(x, initialX, initialY);
        y[0] = Math.max(0.0, y[0]);
        // System.out.println("v=" + y[0] + ", x=" + y[1]);
        y = this.vehiclePhysical.setSpeedAndProgression(y[0], y[1]);
        return y;
    }

    /**
     * Stop the integration; do not make calculations anymore.
     */
    public void stopIntegration()
    {
        try
        {
            super.simulator.removeListener(this,
                    SimulatorInterface.TIME_CHANGED_EVENT);
        } catch (RemoteException e)
        {
            Logger.warning(this, "stopIntegration", e);
        }
    }
}
